/* keylist.c -  key listing
 *	Copyright (C) 2000, 2001 Werner Koch (dd9jn), g10 Code GmbH
 *	Copyright (C) 2002-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <assert.h>
#include <windows.h>

#include "util.h"
#include "context.h"
#include "ops.h"
#include "key.h"

static void finish_key ( gpgme_ctx_t ctx );


static void
keylist_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char *args )
{
    if( ctx->out_of_core )
        return;
    
    switch( code ) {
    case STATUS_EOF:
        if( ctx->tmp_key )
            finish_key( ctx );
        break;
        
    default: /* ignore all other codes */
        break;
    }
}



static time_t
parse_timestamp( char * p )
{    
    return *p ? (time_t)strtoul( p, NULL, 10 ) : 0;
}


static void
set_mainkey_trust_info( gpgme_key_t key, const char * s )
{
    /* look at letters and stop at the first digit */
    for( ; *s && !isdigit( *s ); s++ ) {
        switch( *s ) {
        case 'u': key->gloflags.ultimate = 1; 
	          key->validity = GPGME_VALIDITY_ULTIMATE; break;
	case '-':
	case 'q': key->validity = GPGME_VALIDITY_UNDEFINED; break;
	case 'm': key->validity = GPGME_VALIDITY_MARGINAL; break;
	case 'f': key->validity = GPGME_VALIDITY_FULL; break;
        case 'e': key->keys.flags.expired = 1; break;
        case 'r': key->keys.flags.revoked = 1; 
		  key->validity = GPGME_VALIDITY_REVOKED; break;
        case 'd': key->keys.flags.disabled = 1; break;
        case 'i': key->keys.flags.invalid = 1; break;
        }
    }
}


static void
set_userid_flags (struct user_id_s * u, const char * s)
{
    /* look at letters and stop at the first digit */
    for (; *s && !isdigit (*s); s++) {
        switch( *s ) {
        case 'r': u->flags.revoked  = 1;
	          u->validity = GPGME_VALIDITY_REVOKED; break;
        case 'i': u->flags.invalid  = 1;
		  u->validity = GPGME_VALIDITY_UNDEFINED; break;
        case 'n': u->validity = GPGME_VALIDITY_NEVER; break;
        case 'm': u->validity = GPGME_VALIDITY_MARGINAL; break;
        case 'f': u->validity = GPGME_VALIDITY_FULL; break;
        case 'u': u->validity = GPGME_VALIDITY_ULTIMATE; break;
        }
    }
}

static void
set_subkey_trust_info( struct subkey_s * k, const char * s )
{
    /* look at letters and stop at the first digit */
    for( ; *s && !isdigit (*s); s++ ) {
        switch( *s ) {
          case 'e': k->flags.expired  = 1; break;
          case 'r': k->flags.revoked  = 1; break;
          case 'd': k->flags.disabled = 1; break;
          case 'i': k->flags.invalid  = 1; break;
        }
    }
}

static void
set_mainkey_capability( gpgme_key_t key, const char * s )
{
    for( ; *s ; s++ ) {
        switch( *s ) {
	case 'a': key->keys.flags.can_auth    = 1; break;
        case 'e': key->keys.flags.can_encrypt = 1; break;
        case 's': key->keys.flags.can_sign    = 1; break;
        case 'c': key->keys.flags.can_certify = 1; break;
        case 'E': key->gloflags.can_encrypt   = 1; break;
        case 'S': key->gloflags.can_sign      = 1; break;
        case 'C': key->gloflags.can_certify   = 1; break;
	case 'D': key->gloflags.disabled      = 1; break;
        }
    }
}

static void
set_subkey_capability (struct subkey_s * k, const char * s)
{
    for (; *s; s++) {
        switch (*s) {
	case 'a': k->flags.can_auth    = 1; break;
        case 'e': k->flags.can_encrypt = 1; break;
        case 's': k->flags.can_sign    = 1; break;
        case 'c': k->flags.can_certify = 1; break;
        }
    }
}

static void
set_mainkey_ownertrust( gpgme_key_t key, const char * s )
{
    switch( *s ) {
    case '-': key->otrust = GPGME_TRUST_UNKNOWN; break;
    case 'e': key->otrust = GPGME_TRUST_EXPIRED; break;
    case 'q': key->otrust = GPGME_TRUST_UNDEFINED; break;
    case 'n': key->otrust = GPGME_TRUST_NEVER; break;
    case 'm': key->otrust = GPGME_TRUST_MARGINAL; break;
    case 'f': key->otrust = GPGME_TRUST_FULLY; break;
    case 'u': key->otrust = GPGME_TRUST_ULTIMATE; break;
    default: key->otrust  = GPGME_TRUST_UNKNOWN; break;
    }
}


/* Note: we are allowed to modify line */
static void
keylist_colon_handler( gpgme_ctx_t ctx, char * line )
{    
    enum {
	RT_NONE, 	  
	RT_SIG, 
	RT_UID, 
	RT_SUB, 
	RT_PUB, 
	RT_FPR, 
	RT_SSB, 
	RT_SEC,
	RT_PKD,
    } rectype = RT_NONE;
    gpgme_key_t key = ctx->tmp_key;
    struct subkey_s * sk = NULL;
    struct signature_s * sig = NULL;
    struct subkey_s * s = NULL;
    struct user_id_s * u = NULL;
    struct mpi_s * m = NULL, * mt = NULL;
    const char * trust_info = NULL;
    char * p, * pend;
    int field = 0;
    unsigned long val=0;
    int i;
    
    if( ctx->out_of_core )
        return;
    if( !line ) { /* EOF */
        if( ctx->tmp_key )
            finish_key( ctx );
        return;
    }
    
    for( p = line; p; p = pend ) {
        field++;
        pend = strchr( p, ':' );
        if( pend ) 
            *pend++ = 0;
        
        if( field == 1 ) {
            if( !strcmp( p, "sig" ) && key ) {
                rectype = RT_SIG;
		sig = _gpgme_key_add_signature( key );
		if( !sig ) {
		    ctx->out_of_core = 1;
		    break;
		}
	    }
            else if( !strcmp( p, "uid" ) && key ) {
                rectype = RT_UID;
                key = ctx->tmp_key;
		sig = _gpgme_key_add_signature( key );
		if( !sig ) {
		    ctx->out_of_core = 1;
		    break;
		}
            }
            else if( !strcmp( p, "sub" ) && key ) {
                /* start a new subkey */
                rectype = RT_SUB;
		sk = _gpgme_key_add_subkey( key );
		if( !sk ) {
                    ctx->out_of_core = 1;
                    return;
                }
            }
            else if( !strcmp( p, "ssb" ) && key ) {
                /* start a new secret subkey */
                rectype = RT_SSB;
		sk = _gpgme_key_add_secret_subkey( key );
		if( !sk ) {
                    ctx->out_of_core = 1;
                    return;
                }
            }
            else if( !strcmp( p, "pub" ) ) {
                /* start a new keyblock */
                if( _gpgme_key_new ( &key ) ) {
                    ctx->out_of_core = 1; /* the only kind of error we can get*/
                    return;
                }
                rectype = RT_PUB;
                if ( ctx->tmp_key )
                    finish_key ( ctx );
                assert ( !ctx->tmp_key );
                ctx->tmp_key = key;
		ctx->tmp_i = 0;
		if (ctx->cb.progress)
		    ctx->cb.progress (ctx->cb.progress_value, "keylist", 0, 
				      ctx->cb.progess_tmp,
				      ctx->cb.progress_value_int);
		ctx->cb.progess_tmp++;
            }
            else if( !strcmp ( p, "sec" ) ) {
                /* start a new keyblock */
                if ( _gpgme_key_new_secret( &key ) ) {
                    ctx->out_of_core = 1; /* the only kind of error we can get */
                    return;
                }
                rectype = RT_SEC;
                if( ctx->tmp_key )
                    finish_key ( ctx );
                assert ( !ctx->tmp_key );
                ctx->tmp_key = key;
            }
            else if( !strcmp( p, "fpr" ) && key ) 
                rectype = RT_FPR;
	    else if( !strcmp( p, "pkd" ) && key )
		rectype = RT_PKD;
            else 
                rectype = RT_NONE;
        }
        else if( rectype == RT_PUB || rectype == RT_SEC ) 
	{
            switch( field ) {
            case 2: /* trust info */
                trust_info = p; 
                set_mainkey_trust_info( key, trust_info );
                break;
            case 3: /* key length */
                i = atoi( p ); 
                if( i > 1 ) /* ignore invalid values */
                    key->keys.key_len = i; 
                break;
            case 4: /* pubkey algo */
                i = atoi ( p );
                if( i > 0 && i < 128 )
                    key->keys.key_algo = i;
                break;
            case 5: /* long keyid */
		memset (key->keys.keyid, 0, sizeof key->keys.keyid);
		strncpy (key->keys.keyid, p, DIM (key->keys.keyid)-1);
                break;
            case 6: /* timestamp (1998-02-28) */
                key->keys.timestamp = parse_timestamp( p );
                break;
            case 7: /* valid for n days */
                key->keys.expires = parse_timestamp( p );
                break;
            case 8: /* reserved (LID) */
                break;
            case 9: /* ownertrust */
                set_mainkey_ownertrust( key, p );				
                break;
            case 10: /* not used due to --fixed-list-mode option */
                break;
            case 11: /* signature class  */
                break;
            case 12: /* capabilities */
                set_mainkey_capability( key, p );
                break;
            case 15:
		if (p && * p) 
		{
		    key->gloflags.divert_to_card = 1;
		    memset (key->keys.cardno, 0, sizeof key->keys.cardno);
		    strncpy (key->keys.cardno, p, DIM (key->keys.cardno));
		}
                pend = NULL;  /* we can stop here */
                break;
            }
        }
        else if( (rectype == RT_SUB || rectype== RT_SSB) && sk ) {
            switch( field ) {
            case 2: /* trust info */
                set_subkey_trust_info( sk, p );
                break;
            case 3: /* key length */
                i = atoi( p );
                if ( i > 1 ) /* ignore invalid values */
                    sk->key_len = i; 
                break;
            case 4: /* pubkey algo */
                i = atoi( p );
                if( i > 0 && i < 128 )
                    sk->key_algo = i;
                break;
            case 5: /* long keyid */
                memset (sk->keyid, 0, sizeof sk->keyid);
		strncpy (sk->keyid, p, DIM (sk->keyid)-1);
                break;
            case 6: /* timestamp (1998-02-28) */
                sk->timestamp = parse_timestamp( p );
                break;
            case 7: /* valid for n days */
                sk->expires = parse_timestamp( p );
                break;
            case 8: /* reserved (LID) */
                break;
            case 9: /* ownertrust */
                break;
            case 10:/* user ID n/a for a subkey */
                break;
            case 11:  /* signature class  */
                break;
            case 12: /* capability */
                set_subkey_capability ( sk, p );
                break;
            case 15:
		memset (sk->cardno, 0, sizeof sk->cardno);
		strncpy (sk->cardno, p, DIM (sk->cardno)-1);
                pend = NULL;  /* we can stop here */
                break;
            }
        }
        else if (rectype == RT_UID) 
	{
            switch (field) 
	    {
            case 2: /* trust info */
                trust_info = p;  /* save for later use */
                break;
	    case 6: /* creation date */
		if (p)
		    val = parse_timestamp (p);
		break;

	    case 8: /* hash over name */
		break;

            case 10: /* user ID */
		safe_free (ctx->tmp_id);
		ctx->tmp_id = strdup (p);
		if (!ctx->tmp_id) {
		    ctx->out_of_core = 1;
		    break;
		}
                if (_gpgme_key_append_name (key, p, &u))
                    ctx->out_of_core = 1;
                else if (trust_info)
		    set_userid_flags (u, trust_info);
		if (sig) {
		    safe_free (sig->issuer);
		    sig->issuer = calloc (1, strlen (p) + 1);
		    if (!sig->issuer) {
			ctx->out_of_core = 1;
			break;
		    }
		    _gpgme_decode_c_string (p, &sig->issuer, strlen (p)+ 1);
		}
		u->created = val;
                pend = NULL;  /* we can stop here */
                break;
            }
        }
        else if( rectype == RT_FPR ) {
	    int nfpr = ctx->tmp_i;
            switch( field ) {
            case 10:
		for( s = &key->keys; s && nfpr; s = s->next, nfpr-- )
		    ;
		if( s ) {
		    s->fingerprint = strdup( p );
		    if( !s->fingerprint ) {
			ctx->out_of_core = 1;
			break;
		    }
		}
		pend = NULL; /* that is all we want */
		ctx->tmp_i++;
		break;
	    }
        }
	else if( rectype == RT_SIG ) {
	    switch( field ) {
	    case 2: /* signature state */
		switch( *p ) {
		case '!': sig->stat = GPGME_SIG_STAT_GOOD;  break;
		case '-': sig->stat = GPGME_SIG_STAT_BAD;   break;
		case '?': sig->stat = GPGME_SIG_STAT_NOKEY; break;
		case '%': sig->stat = GPGME_SIG_STAT_ERROR; break;
		default : sig->stat = GPGME_SIG_STAT_NONE;  break;
		}
		break;

	    case 4: /* pk algorithm */
		sig->algo = atol (p);
		break;

	    case 5: /* long keyid */
		memset (sig->keyid, 0, sizeof sig->keyid);
		strncpy (sig->keyid, p, DIM (sig->keyid)-1);
		break;

	    case 6: /* created */
		sig->created = parse_timestamp( p );
		break;

	    case 7: /* expires */
		sig->expires = parse_timestamp( p );
		break;

	    case 10:
		if( p ) {
		    sig->issuer = calloc( 1, strlen( p ) + 1 );
		    if( !sig->issuer ) {
			ctx->out_of_core = 1;
			break;
		    }
		    _gpgme_decode_c_string( p, &sig->issuer, strlen( p ) + 1 );
		}
		break;

	    case 11:
		sig->sigclass = strtoul( p, NULL, 16 );
		if (strchr (p, 'l'))
		    sig->is_local = 1;
		/*if (strchr (p, ' '))
		    sig->is_nonrev = 1;*/
		if (sig->sigclass < 0x10 || sig->sigclass > 0x13) {
		    /* we only want certificates and no key signatures */
		    pend = NULL;
		    break;
		}
		if( ctx->tmp_id ) {
		    sig->user_id = calloc( 1, strlen( ctx->tmp_id ) + 1 );
		    if( !sig->user_id ) {		    
			ctx->out_of_core = 1;
			break;
		    }
		    _gpgme_decode_c_string( ctx->tmp_id, &sig->user_id, strlen( ctx->tmp_id ) +1 );
		}
		pend = NULL;  /* we can stop here */
		break;
	    }
	}
	else if( rectype == RT_PKD ) {
	    switch( field ) {
	    case 1:
		m = calloc( 1, sizeof * m );
		if( !m ) {
		    ctx->out_of_core = 1;
		    return;
		}
		for( mt=key->pkey; mt->next; mt=mt->next )
		    ;
		mt->next = m;
		break;

	    case 2:
		if( m )
		    m->bits = atol( p );
		break;

	    case 3:
		if( m ) {
		    m->hexval = strdup( p );
		    if( !m->hexval )
			ctx->out_of_core=1;
		}
		break;
	    }
	}
    }
    
}


/*
 * We have read an entire key into ctx->tmp_key and should now finish
 * it.  It is assumed that this releases ctx->tmp_key.
 */
static void
finish_key( gpgme_ctx_t ctx )
{
    gpgme_key_t key = ctx->tmp_key;
    struct key_queue_item_s * q, * q2;
    
    assert( key );
    ctx->tmp_key = NULL;
    
    q = malloc( sizeof *q );
    if( !q ) {
        gpgme_key_release( key );
        ctx->out_of_core = 1;
        return;
    }
    q->key = key;
    q->next = NULL;
    /* fixme: lock queue. Use a tail pointer? */
    q2 = ctx->key_queue;
    if( !q2 )
        ctx->key_queue = q;
    else {
        for( ; q2->next; q2 = q2->next )
            ;
        q2->next = q;
    }
    ctx->key_cond = 1;
    /* fixme: unlock queue */
}



/**
 * gpgme_op_keylist_start:
 * @c: context 
 * @pattern: a GnuPg user ID or NULL for all
 * @secret_only: List only keys where the secret part is available
 * 
 * Note that this function also cancels a pending key listing operaton..
 * 
 * Return value:  0 on success or an errorcode. 
 **/
gpgme_error_t
gpgme_op_keylist_start( gpgme_ctx_t ctx,  const char *pattern, int secret_only )
{
    gpgme_error_t rc = 0;
    
    if( !ctx )
        return mk_error( Invalid_Value );
    ctx->pending = 1;
    
    _gpgme_release_result( ctx );
    ctx->out_of_core = 0;
    ctx->cb.progess_tmp = 0;

    _gpgme_gpg_release( &ctx->gpg );           
    gpgme_key_release( ctx->tmp_key );
    ctx->tmp_key = NULL;
    /* Fixme: release key_queue */
    
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;
    
    _gpgme_gpg_set_status_handler( ctx->gpg, keylist_status_handler, ctx );
    rc = _gpgme_gpg_set_colon_line_handler( ctx->gpg, keylist_colon_handler, ctx );
    if ( rc )
        goto leave;
    
    /* build the commandline */
    if (ctx->use_logging)
	_gpgme_gpg_set_logging_handler (ctx->gpg, ctx);
    _gpgme_gpg_set_list_options (ctx->gpg, ctx->list_opts);
    _gpgme_gpg_add_arg( ctx->gpg, "--with-colons" );
    _gpgme_gpg_add_arg( ctx->gpg, "--fixed-list-mode" );
    _gpgme_gpg_add_arg( ctx->gpg, "--with-fingerprint" );
    _gpgme_gpg_add_arg( ctx->gpg, "--with-fingerprint" ); /* for the subkey */
    if( ctx->keylist_mode & 1 )
        _gpgme_gpg_add_arg( ctx->gpg, "--no-expensive-trust-checks" );
    if( ctx->keylist_mode & 2 )
	_gpgme_gpg_add_arg( ctx->gpg, "--with-key-data" );
    if( ctx->keylist_mode & 4 )
	_gpgme_gpg_add_arg( ctx->gpg, "--fast-list-mode" );
    if( 1 || !ctx->keylist_mode ) /* XXX: do all combinations make sense? */
	_gpgme_gpg_add_arg( ctx->gpg, "-vvv" ); /* to list and check sigs */
    _gpgme_gpg_add_arg ( ctx->gpg, secret_only?
                         "--list-secret-keys": /*"--list-keys"*/ "-k" );
    
    /* Tell the gpg object about the data */
    _gpgme_gpg_add_arg ( ctx->gpg, "--" );
    if( pattern && *pattern )
        _gpgme_gpg_add_arg ( ctx->gpg, pattern );
    
    /* and kick off the process */
    rc = _gpgme_gpg_spawn ( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release ( &ctx->gpg );
    }
    return rc;
}


gpgme_error_t
gpgme_op_keylist_next( gpgme_ctx_t ctx, gpgme_key_t * r_key )
{
    struct key_queue_item_s * q;
    
    if( !r_key )
        return mk_error( Invalid_Value );
    *r_key = NULL;
    if( !ctx )
        return mk_error( Invalid_Value );
    if( !ctx->pending )
        return mk_error( No_Request );
    if( ctx->out_of_core )
        return mk_error( Out_Of_Core );
    
    if( !ctx->key_queue ) {
        _gpgme_wait_on_condition( ctx, 1, &ctx->key_cond );
        if ( ctx->out_of_core )
            return mk_error( Out_Of_Core );
        if( !ctx->key_cond )
            return mk_error( EOF );
        ctx->key_cond = 0; 
        assert ( ctx->key_queue );
    }
    q = ctx->key_queue;
    ctx->key_queue = q->next;
    
    *r_key = q->key;
    safe_free( q );
    return 0;
}
