/* wptPassCache.c - keep a cache of passphrases
 *	Copyright (C) 2002 Free Software Foundation, Inc.
 *	Copyright (C) 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>

#include "wptTypes.h"
#include "wptErrors.h"
#include "wptAgent.h"


struct secret_data_s {
    int  totallen; /* this includes the padding */
    int  datalen;  /* actual data length */
    char data[1];
};

typedef struct cache_item_s *ITEM;
struct cache_item_s {
    ITEM next;
    time_t created;
    time_t accessed;
    int  ttl;  /* max. lifetime given in seconds */
    int lockcount;
    struct secret_data_s * pw;
    size_t pwlen;
    char key[1];
};


static ITEM thecache;


static void
release_data( struct secret_data_s * data )
{
    free( data );
}


static void
wipe_data( void * data, size_t dlen )
{
    memset( data, 0xff, dlen );
    memset( data, 0xaa, dlen );
    memset( data, 0x55, dlen );
    memset( data, 0x00, dlen );
}


static struct secret_data_s *
new_data( const void * data, size_t length )
{
    struct secret_data_s * d;
    int total;

    /* we pad the data to 32 bytes so that it get more complicated
       finding something out by watching allocation patterns.  This is
       usally not possible but we better assume nothing about our
       secure storage provider*/
    total = length + 32 - (length % 32);

    d = (secret_data_s *)malloc( sizeof *d + total - 1 );
    if( d ) {
        d->totallen = total;
        d->datalen  = length;
        memcpy( d->data, data, length );
    }
    return d;
}


/* check whether there are items to expire */
static void
housekeeping( void )
{
    ITEM r, rprev;
    time_t current = time( NULL );

    /* first expire the actual data */
    for( r=thecache; r; r = r->next ) {
        if( !r->lockcount && r->pw && r->accessed + r->ttl < current ) {
	    wipe_data( r->pw, r->pwlen );
            release_data( r->pw );
            r->pw = NULL;
            r->accessed = current;
        }
    }

    /* second, make sure that we also remove them based on the created stamp so
       that the user has to enter it from time to time.  We do this every hour */
    for( r=thecache; r; r = r->next ) {
        if( !r->lockcount && r->pw && r->created + 60*60 < current ) {
	    wipe_data( r->pw, r->pwlen );
            release_data( r->pw );
            r->pw = NULL;
            r->accessed = current;
        }
    }

    /* third, make sure that we don't have too many items in the list.
       Expire old and unused entries after 30 minutes */
    for( rprev=NULL, r=thecache; r; ) {
        if (!r->pw && r->accessed + 60*30 < current) {
            if (r->lockcount) {
		BUG( NULL );
                r->accessed += 60*10; /* next error message in 10 minutes */
                rprev = r;
                r = r->next;
            }
            else {
                ITEM r2 = r->next;
                free (r);
                if( !rprev )
                    thecache = r2;
                else
                    rprev->next = r2;
                r = r2;
            }
        }
        else {
            rprev = r;
            r = r->next;
        }
    }
}


void
agent_flush_cache( void )
{
    ITEM r;    

    for( r=thecache; r; r = r->next ) {
        if( !r->lockcount && r->pw ) {
	    wipe_data( r->pw, r->pwlen );
            release_data( r->pw );
            r->pw = NULL;
            r->accessed = 0;
        }
        else if( r->lockcount && r->pw ) {
            r->accessed = 0;
            r->ttl = 0;
        }
    }
}


/* Try to find the item given by key and set the TTL value to zero.
   This means the item expires the next time the passphrase cache is used. */
int
agent_del_cache( const char * key )
{
    void * item;
    ITEM r;

    if( agent_get_cache( key, &item ) ) {
	r = (ITEM)item;
	if( r )
	    r->ttl = 0;
	agent_unlock_cache_entry( &item );
	return 0;
    }
    return -1;
}


/* Store DATA of length DATALEN in the cache under KEY and mark it
   with a maximum lifetime of TTL seconds.  If there is already data
   under this key, it will be replaced.  Using a DATA of NULL deletes
   the entry */
int
agent_put_cache( const char * key, const char * data, int ttl )
{
    ITEM r;
    
    housekeeping ();

    if (ttl < 1)
        ttl = 5*60;
    if (!ttl)
        return 0;

    for (r=thecache; r; r = r->next) {
        if (!r->lockcount && !strcmp (r->key, key))
            break;
    }
    if( r ) { /* replace */
        if( r->pw ) {
	    wipe_data( r->pw, r->pwlen );
            release_data( r->pw );
            r->pw = NULL;
        }
        if (data) {
            r->created = r->accessed = time( NULL );
            r->ttl = ttl;
	    r->pwlen = strlen( data );
            r->pw = new_data( data, r->pwlen+1 );
            if (!r->pw)
                BUG( NULL );
        }
    }
    else if (data) { /* simply insert */
        r = (ITEM)calloc (1, sizeof *r + strlen (key));
        if (!r)
            BUG( NULL );
        else {
            strcpy (r->key, key);
            r->created = r->accessed = time( NULL ); 
            r->ttl = ttl;
	    r->pwlen = strlen( data );
            r->pw = new_data( data, r->pwlen+1 );
            if (!r->pw)
                BUG( NULL );
            else {
                r->next = thecache;
                thecache = r;
            }
        }
    }
    return 0;
}


/* Try to find an item in the cache */
const char *
agent_get_cache( const char * key, void ** cache_id )
{
    ITEM r;
    
    housekeeping ();

    /* first try to find one with no locks - this is an updated cache
       entry: We might have entries with a lockcount and without a
       lockcount. */
    for (r=thecache; r; r = r->next) {
        if (!r->lockcount && r->pw && !strcmp (r->key, key)) {
            /* put_cache does only put strings into the cache, so we
               don't need the lengths */
            r->accessed = time (NULL);
            r->lockcount++;
            *cache_id = r;
            return r->pw->data;
        }
    }
    /* again, but this time get even one with a lockcount set */
    for (r=thecache; r; r = r->next) {
        if (r->pw && !strcmp (r->key, key)) {
            r->accessed = time (NULL);
            r->lockcount++;
            *cache_id = r;
            return r->pw->data;
        }
    }    

    *cache_id = NULL;
    return NULL;
}


void
agent_unlock_cache_entry( void ** cache_id )
{
    ITEM r;

    for( r=thecache; r; r = r->next ) {
        if( r == *cache_id ) {
            if( !r->lockcount )
		BUG( NULL );
            else
                r->lockcount--;
            return;
        }
    }
}
