/* wptWipeFile.c - Secure file removal
 *	Copyright (C) 2000 Matt Gauthier
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * WinPT software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 **************************************************************************
 * This code based on the sunlink.c file from the SRM project, but        *
 * it was heavily modified to work with W32 and with added GCRYPT         *
 * support for gathering random bytes.                                    *
 *                                                                        *
 * The original code was placed under the GNU Lesser Public License,      *
 * even so I decide to put this file under the GNU General Public License.*
 **************************************************************************
 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include <direct.h>

#include "openpgp.h"

enum {
    WIPE_MODE_SIMPLE  = 0,
    WIPE_MODE_DOD     = 1,
    WIPE_MODE_GUTMANN = 2
};

enum { /* WinPT error codes */
    ERR_FILE_READ   =  4,
    ERR_FILE_OPEN   =  2,
    ERR_FILE_REMOVE =  8,
    ERR_FILE_ZERO   =  9,
    ERR_NOMEMORY    = 25
};

typedef unsigned __int64 DDWORD;

typedef struct {    
    HANDLE fd;
    DDWORD filesize;
    byte * buffer;
    size_t buffsize;
} wipe_context_s;


void (*progress_cb) (void *, DDWORD, DDWORD);
static void * progress_cb_value = NULL;


static void 
overwrite (wipe_context_s * ctx)
{
    DDWORD blocks = 0, mod = 0;
    DWORD nwritten = 0, npos = 0;
    DWORD size_high = 0;
    
    blocks = ctx->filesize / ctx->buffsize;
    mod = ctx->filesize % ctx->buffsize;
    SetFilePointer (ctx->fd, 0, &size_high, FILE_BEGIN);
    while (blocks--) {	
	if (!WriteFile (ctx->fd, ctx->buffer, ctx->buffsize, &nwritten, NULL))
	    break;
	npos += nwritten;
	if (progress_cb)
	    progress_cb (progress_cb_value, npos, ctx->filesize);
    }
    if (mod) {
	WriteFile (ctx->fd, ctx->buffer, (DWORD)mod, &nwritten, NULL);
	npos += nwritten;
	if (progress_cb)
	    progress_cb (progress_cb_value, npos, ctx->filesize);
    }
    FlushFileBuffers (ctx->fd);
    SetFilePointer (ctx->fd, 0, &size_high, FILE_BEGIN);
} /* overwrite */


static void
randomize_buffer (byte * buf, size_t bufsize, int level)
{
    const int blocksize = 512;
    int blocks = bufsize / blocksize;
    int mod = bufsize % blocksize;
    
    while (blocks--) {
	gpg_randomize (buf, blocksize, level);
	buf += blocksize;
    }
    if (mod)
	gpg_randomize (buf, mod, level);
} /* randomize_buffer */


static void 
overwrite_random (int num_passes, wipe_context_s * ctx) 
{  	
    int i = 0;
    
    for (i = 0; i < num_passes; i++) {
	randomize_buffer (ctx->buffer, ctx->buffsize, 0);
	overwrite (ctx);
    }
} /* overwrite_random */


static void 
overwrite_byte (int byte, wipe_context_s * ctx) 
{
    memset (ctx->buffer, byte, ctx->buffsize);
    overwrite (ctx);
} /* overwrite_byte */


static void
overwrite_bytes (int byte1, int byte2, int byte3, wipe_context_s * ctx)
{
    size_t i = 0;

    memset (ctx->buffer, byte1, ctx->buffsize);
    for (i = 1; i < ctx->buffsize; i += 3) {
	ctx->buffer[i] = byte2;
	ctx->buffer[i+1] = byte3;
    }
    overwrite (ctx);
} /* overwrite_bytes */


/**
 * For the case the file is not a regular file (this is true for
 * devices or directories) this function tries to rename the file
 * to random pattern and then it will be delete (without random!). 
 **/
int 
rename_unlink (const char * path)
{
    struct stat statbuf;
    char * new_name = NULL, * p = NULL, c;  
    int i = 0, rc = 0;
    int is_dir = 0;
    
    
    if (GetFileAttributes (path) & FILE_ATTRIBUTE_DIRECTORY)
	is_dir = 1;
    
    new_name = (char *)malloc (strlen (path) + 15);
    if (!new_name)
	return ERR_NOMEMORY;
    
    strcpy (new_name, path);
    p = strrchr (new_name, '\\');
    if (p != NULL) {
	p++;
	*p = '\0';
    } 
    else
	p = new_name;
    do {
	while (i < 14) {
	    c = gpg_random_char (1);
	    *p = c; p++; i++;
	}
	*p = '\0';
    } while (stat (new_name, &statbuf) == 0);
    
    if (rename (path, new_name) == -1) {
	rc = ERR_FILE_READ;
	goto leave;
    }
    if (is_dir && RemoveDirectory (new_name) == FALSE)
	rc = ERR_FILE_REMOVE;
    else if (!DeleteFile (new_name))
	rc = ERR_FILE_REMOVE;
    
leave:
    free (new_name);
    return rc;
} /* rename_unlink */


static __int64
GetFileSize64( const char * path )
{
    FILE *fp = fopen( path, "r" );
    if( fp ) {
	struct _stati64 statbuf;
	_fstati64( fileno( fp ), &statbuf );
	fclose( fp );
	return statbuf.st_size;
    }
    return -1;
} /* GetFileSize64 */


int 
secure_unlink( const char *path, const int mode ) 
{	
    wipe_context_s ctx = {0};
    DWORD size_high = 0;
    
    if( GetFileAttributes( path ) & FILE_ATTRIBUTE_DIRECTORY)
	return rename_unlink( path );
    
    ctx.buffsize = 8192;
    ctx.buffer = (byte *)malloc( ctx.buffsize );
    if( !ctx.buffer )
	return ERR_NOMEMORY;
    
    ctx.filesize = GetFileSize64( path );
    if( !ctx.filesize ) {
	free( ctx.buffer );
	return ERR_FILE_ZERO;
    }

    ctx.fd = CreateFile( path, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, 0, NULL );
    if( ctx.fd == INVALID_HANDLE_VALUE ) {
	free( ctx.buffer );
	return ERR_FILE_OPEN;
    }
    
    gpg_quick_random_gen( 1 );

    switch( mode ) {
    case WIPE_MODE_SIMPLE:
	overwrite_random( 2, &ctx );
	break;
        
    case WIPE_MODE_DOD:
	overwrite_random( 1, &ctx );
	overwrite_byte( (~1) & 0xFF, &ctx );
	overwrite_random( 1, &ctx );
	overwrite_byte( (~4) & 0xFF, &ctx );
	overwrite_random( 1, &ctx );
	break;
        
    case WIPE_MODE_GUTMANN:
	overwrite_random( 4, &ctx);
	overwrite_byte( 0x55, &ctx );
	overwrite_byte ( 0xAA, &ctx );
	overwrite_bytes( 0x92, 0x49, 0x24, &ctx );
	overwrite_bytes( 0x49, 0x24, 0x92, &ctx );
	overwrite_bytes( 0x24, 0x92, 0x49, &ctx );
	overwrite_byte( 0x00, &ctx );
	overwrite_byte( 0x11, &ctx );
	overwrite_byte( 0x22, &ctx );
	overwrite_byte( 0x33, &ctx );
	overwrite_byte( 0x44, &ctx );
	overwrite_byte( 0x55, &ctx );
	overwrite_byte( 0x66, &ctx );
	overwrite_byte( 0x77, &ctx );
	overwrite_byte( 0x88, &ctx );
	overwrite_byte( 0x99, &ctx );
	overwrite_byte( 0xAA, &ctx );
	overwrite_byte( 0xBB, &ctx );
	overwrite_byte( 0xCC, &ctx );
	overwrite_byte( 0xDD, &ctx );
	overwrite_byte( 0xEE, &ctx );
	overwrite_byte( 0xFF, &ctx );
	overwrite_bytes( 0x92, 0x49, 0x24, &ctx );
	overwrite_bytes( 0x49, 0x24, 0x92, &ctx );
	overwrite_bytes( 0x24, 0x92, 0x49, &ctx );
	overwrite_bytes( 0x6D, 0xB6, 0xDB, &ctx );
	overwrite_bytes( 0xB6, 0xDB, 0x6D, &ctx );
	overwrite_bytes( 0xDB, 0x6D, 0xB6, &ctx );
	overwrite_random( 4, &ctx );
	break;
    }
        
    /* Set file length to zero so allocated clusters cannot be trailed */	
    SetFilePointer( ctx.fd, 0, &size_high, FILE_BEGIN );
    SetEndOfFile( ctx.fd );
    CloseHandle( ctx.fd );
    
    /* free memory */
    memset( ctx.buffer, 0, ctx.buffsize ); /* burn the last evidence */	
    free( ctx.buffer);
    
    return rename_unlink( path );
} /* secure_unlink */


/* Windows 98 - Q188074 */
#define REGISTRY_FILESYSTEM "System\\CurrentControlSet\\Control\\FileSystem"
#define REGISTRY_LOWDISKSPACE "DisableLowDiskSpaceBroadcast"


/* disables the annoying warning Windows 98 displays when disk space is low */
static void
handle_lowdiskspace_notify( const char * drive, int disable )
{    
    OSVERSIONINFO ov;
    HKEY key;
    DWORD n;

    memset( &ov, 0, sizeof ov );
    ov.dwOSVersionInfoSize = sizeof ov;
    GetVersionEx( &ov );
    if( ov.dwPlatformId == VER_PLATFORM_WIN32_NT )
	return;

    if( disable ) {
	unsigned new = (1 << (toupper((unsigned)drive) - (unsigned)'A'));
	if( RegOpenKey( HKEY_LOCAL_MACHINE, REGISTRY_FILESYSTEM, &key ) ) {
	    n = sizeof new;
	    RegSetValue( key, REGISTRY_LOWDISKSPACE, REG_DWORD, (LPCTSTR)new, n );
	    RegCloseKey( key );
        }
    }
    else {
	if( RegOpenKey( HKEY_LOCAL_MACHINE, REGISTRY_FILESYSTEM, &key ) ) {
	    RegDeleteKey( key, REGISTRY_LOWDISKSPACE );
	    RegCloseKey( key );
	}
    }
} /* handle_lowdiskspace_notify */


int
wipe_freespace( const char * drive, 
	        void (*cb)(void *, DDWORD, DDWORD), void * cb_value )
{
    ULARGE_INTEGER caller, total, frees;
    HANDLE fd;
    int disktyp = GetDriveType( drive ), rc;
    char * file;
    OSVERSIONINFO osinf;

    memset (&osinf, 0, sizeof osinf);
    osinf.dwOSVersionInfoSize = sizeof osinf;
    GetVersionEx (&osinf);    

    if( disktyp != DRIVE_FIXED && disktyp != DRIVE_REMOVABLE )
	return -1;
    if( !GetDiskFreeSpaceEx( drive, &caller, &total, &frees ) )
	return -1;
    if (osinf.dwPlatformId != VER_PLATFORM_WIN32_NT)
	handle_lowdiskspace_notify (drive, 1);

    if( !frees.LowPart )
	return ERR_FILE_ZERO;
    file = malloc( strlen( drive ) + 8 );
    if( !file )
	return ERR_NOMEMORY;
    sprintf( file, "%stemp", drive );

    fd = CreateFile( file,
		    GENERIC_READ|GENERIC_WRITE,
		    FILE_SHARE_READ|FILE_SHARE_WRITE,
		    NULL, CREATE_ALWAYS, 0, NULL );
    if( fd == INVALID_HANDLE_VALUE ) {
	free( file );
	return ERR_FILE_OPEN;
    }
    SetFilePointer( fd, frees.LowPart, &frees.HighPart, FILE_BEGIN );
    SetEndOfFile( fd );
    CloseHandle( fd );

    if( cb && cb_value ) {
	progress_cb = cb;
	progress_cb_value = cb_value;
    }
    rc = secure_unlink( file, WIPE_MODE_SIMPLE );

    handle_lowdiskspace_notify( drive, 0 );
    free( file );
    return rc;
} /* wipe_freespace */