# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

import sys, types
from PyQt4 import QtCore, QtGui
import widgets, stylesheets, objects

class GuiCell(QtCore.QObject):
    """This class is a stand-in for the real cell class

    This is a layer of abstraction between the actual cell calculations and the
    widgets that display it. Also it includes orientation as a cell value.
    """
    def __init__(self, name, func, value, orientation, abs_ref,
                 include_col_form = True):
        QtCore.QObject.__init__(self)
        self.name, self.func, self.value = name, func, value
        self.orientation = orientation
        self.include_col_form = include_col_form
        self.value_widget = None
        self.abs_ref = abs_ref
        self.colform_text = ''

    def isparent(self):
        return (type(self.value) in (types.ListType, types.TupleType) or
                isinstance(self.value, GuiTable))

    def istable(self):
        return isinstance(self.value, GuiTable)

    def set_colform(self, property_set):
        """Set the column formula string to the given text"""
        self.colform_text = str(property_set.form_str)

    def editingFinished(self, comptype, text):
        """Call this (eventually) when editing finished"""
        if ((comptype == 'name' and text != self.name) or
            (comptype == 'func' and text != self.func)):
            print "GuiCell: valueChanged:", comptype, self.abs_ref, text
            self.emit(QtCore.SIGNAL('valueChanged'), comptype,
                      self.abs_ref, text)
        elif comptype == 'colform' and text != self.colform_text:
            print "GuiCell: Column formula update in ", self.abs_ref
            self.emit(QtCore.SIGNAL('valueChanged'), 'func',
                      self.abs_ref.append('C'), text)
        else:
            print "GuiCell: editingFinished:", comptype, self.abs_ref
            self.emit(QtCore.SIGNAL('editingFinished'), comptype, self.abs_ref)

    def table_signal_handler(self, comptype, axis, index, text):
        """Called when editing finished in a table"""
        assert comptype in ('name', 'func'), comptype
        if index == 0: ar_suffix = (axis,)
        elif axis == 'R': ar_suffix = (index, 0)
        elif axis == 'C': ar_suffix = (0, index)
        else: assert False, (axis, index)
        ar = self.abs_ref.append(ar_suffix)

        # Right now, just assume value always changed if any edits
        print "GuiCell: valueChanged:", ar, text
        self.emit(QtCore.SIGNAL('valueChanged'), comptype, ar, text)

    def signal_handler(self, comptype, index, text):
        """Called on editingFinished signal"""
        if index == 'C': # Column formula edited
            if comptype == 'func':
                self.editingFinished('colform', text)
            else: pass # this option only possible because of extra widget
        else: # Route message to appropriate GuiCell
            self.value[index].editingFinished(comptype, text)

    def set_widget_table_recursive(self):
        """Make a table widget describing value and set as self.value_widget"""
        assert self.isparent(), (self.name, self.func)
        if not self.istable():
            for subcell in self.value:
                if subcell.isparent(): subcell.set_widget_table_recursive()
        self.set_value_widget()

    def flip_orientation(self):
        """Go from tall to wide or vice versa, rewriting widget"""
        if self.orientation == 'tall': self.orientation = 'wide'
        else: self.orientation = 'tall'
        self.set_value_widget()

    def set_value_widget(self):
        """Create the value widget and bind signals"""
        if self.istable():
            self.value_widget = self.value.get_widget()
            self.connect(self.value_widget, QtCore.SIGNAL('editingFinished'),
                         self.table_signal_handler)
        else:
            self.value_widget = List(self.orientation, self.include_col_form)
            self.value_widget.add_cells(self.value)
            if self.include_col_form:
                self.value_widget.set_colform_text(self.colform_text)
            self.connect(self.value_widget, QtCore.SIGNAL('editingFinished'),
                         self.signal_handler)

    def get_ref(self, ar):
        """Return cell corresponding to reference"""
        if len(ar) == 0: return self
        assert self.isparent(), self
        return self.value[ar.car()].get_ref(ar.cdr())

    def isroot(self): return self.abs_ref.isroot()

    def redraw(self, child_cell):
        """Reread and redraw the widget corresponding to child_cell"""
        self.value_widget.redraw(child_cell)


class GuiTable(QtCore.QObject):
    """This class is a stand-in for the ExTable backend class
    
    Like GuiCell, this object is in-between a widget (Table2D) and the actual
    data structure (ExTable).
    """
    def __init__(self, prog, extable, ar):
        """Initialize with ExTable object
        
        extable will only be used to set initial values and will not be written
        to.
        """
        QtCore.QObject.__init__(self)
        self.prog, self.ar = prog, ar
        self.extable = extable
        self.height = self.extable.get_num_rows()
        self.width = self.extable.get_num_cols()

    def get_widget(self):
        """Convert the extable to a Table2D Widget"""
        w = widgets.Table2D(self.height, self.width)
        self.set_values(w)
        self.set_rowcol_names(w)
        self.set_formulas(w)
        return w
    
    def to_string(self, val):
        """Convert a possibly ExObject to a string, or None (kludge)"""
        if isinstance(val, objects.ExObject):
            assert val.isscalar(), val
            if val.t == 'blank': return None
            return str(val.obj)
        return str(val)

    def set_values(self, table2d):
        """Set the state of the value widgets"""
        for i in range(self.height):
            for j in range(self.width):
                value = self.extable.get_value_thunk(i,j)()
                val_str = self.to_string(value)
                if val_str:
                    table2d.set_value_text(i, j, val_str)

    def set_rowcol_names(self, table2d):
        """Set the names in the rows and columns"""
        table2d.set_axis_names(self.extable.row_axis_name,
                               self.extable.col_axis_name)
        for i in range(self.height):
            rowname = self.to_string(self.extable.get_row_name(i))
            if rowname:
                table2d.set_rowname_text(i, rowname)
        for j in range(self.width):
            colname = self.to_string(self.extable.get_col_name(j))
            if colname:
                table2d.set_colname_text(j, colname)

    def set_formulas(self, table2d):
        """Set the formula text in Table2D widget"""
        row_ps, col_ps = self.prog.get_table_axis_propsets(self.ar)
        row_form_text = row_ps.form_str if row_ps and row_ps.form_str else ''
        col_form_text = col_ps.form_str if col_ps and col_ps.form_str else ''
        table2d.set_axis_formulas(row_form_text, col_form_text)

        for j in range(self.width):
            col_ps = self.prog.get_table_colname_ps(self.ar, j+1)
            if col_ps and col_ps.form_str:
                table2d.set_colform_text(j, col_ps.form_str)


class List(QtGui.QWidget):
    """Draw a table of cells

    A table is composed (visually and in terms of program objects) of
    a number of ListSection (which hold the basic cells) possibly
    interleaved with NameFunc widgets, which hold arrays.
    """
    align = QtCore.Qt.AlignTop | QtCore.Qt.AlignLeft
    paint_offset = 20
    def __init__(self, orientation, include_col_form = True):
        QtGui.QWidget.__init__(self)
        self.orientation = orientation
        self.include_col_form = include_col_form
        self.makeLayout()

    def paintEvent(self, event):
        """Draw the line connecting the various sections"""
        def get_painter():
            painter = QtGui.QPainter(self)
            painter.setRenderHint(QtGui.QPainter.Antialiasing, False)
            color = QtGui.QColor(stylesheets.name_qcolor)
            pen = QtGui.QPen(color, 3, QtCore.Qt.SolidLine,
                             QtCore.Qt.SquareCap) 
            painter.setPen(pen)
            return painter

        def get_coords():
            """Return tuple of line coords: (startx, starty, endx, endy)"""
            top_left_qp = self.widget_tuples[0][0].geometry().topLeft()
            start_coords = (top_left_qp.x() + self.paint_offset,
                            top_left_qp.y() + self.paint_offset)
            last_widget_geom = self.widget_tuples[-1][0].geometry()

            if self.orientation == 'tall':
                bottom_left_qp = last_widget_geom.bottomLeft()
                end_coords = (bottom_left_qp.x() + self.paint_offset,
                              bottom_left_qp.y() - self.paint_offset)
            else:
                top_right_qp = last_widget_geom.topRight()
                end_coords = (top_right_qp.x() - self.paint_offset,
                              top_right_qp.y() + self.paint_offset)
            return start_coords+end_coords

        if self.widget_tuples:
            get_painter().drawLine(*get_coords())

    def add_cells(self, celllist):
        """Convert the given GuiCells to widgets and lay them out"""
        self.celllist = celllist
        self.set_widget_tuples(self.separate_celllist())
        self.add_widgets()

    def get_tablesection(self, celllist, begin, end):
        """Return a ListSection holding the given cells"""
        ts = widgets.ListSection(self.orientation, len(celllist),
                                      begin, end)
        ts.setText([(cell.name, cell.func, cell.value) for cell in celllist])
        return ts

    def emit_helper(self, index):
        def helper(comptype, i, text):
            """Take the NameFunc signal and reraise with cell index"""
            print "List: editingFinished:", comptype, index+i, text
            self.emit(QtCore.SIGNAL('editingFinished'),
                      comptype, index+i, text)
        return helper

    def get_namefunc(self, cell, index):
        """Return a sublayout for the complex cell and bind the signals"""
        assert cell.isparent(), cell
        if self.orientation == 'tall':
            layout = widgets.NameFuncHLayout(cell.name, cell.func,
                                             cell.value_widget)
        else: layout = widgets.NameFuncVLayout(cell.name, cell.func,
                                               cell.value_widget)
        self.connect(layout.nf, QtCore.SIGNAL('editingFinished'),
                     self.emit_helper(index))
        return layout

    def add_column_formula(self):
        """Add the column formula widget to the layout"""
        def colform_emit_helper(comptype, i, text):
            """Take the NameFunc signal and reraise with index = 'C'"""
            self.emit(QtCore.SIGNAL('editingFinished'), comptype, 'C', text)
        w = widgets.NameFunc(self.orientation)
        self.layout.addWidget(w, 0, QtCore.Qt.AlignTop | QtCore.Qt.AlignLeft)
        self.connect(w, QtCore.SIGNAL('editingFinished'), colform_emit_helper)
        self.colform_w = w

    def set_colform_text(self, text):
        """Set the text of the column formula"""
        self.colform_w.setText('', text)

    def add_widgets(self):
        """Add and bind the widgets in self.widget_tuples"""
        if self.include_col_form: self.add_column_formula()
        for obj, index in self.widget_tuples:
            if isinstance(obj, QtGui.QWidget): # ListSection
                self.layout.addWidget(obj, 0, self.align)
                self.connect(obj, QtCore.SIGNAL('editingFinished'),
                             self.emit_helper(index))
            else: # Namefunc layout
                assert isinstance(obj, QtGui.QLayout), obj
                self.layout.addLayout(obj)
        self.layout.addStretch(1)

    def set_widget_tuples(self, sep_celllist):
        """Set the widget tuple list

        The widget tuple list is a list of tuples (widget or layout,
        cell_index).  If there is more than one cell index in widget,
        enter the first one.
        """
        self.widget_tuples = []
        widgets_seen = 0
        for index, elem in enumerate(sep_celllist):
            if type(elem) is types.ListType:
                at_end = (index == len(sep_celllist)-1)
                at_begin = (widgets_seen == 0)
                widget = self.get_tablesection(elem, at_begin, at_end)
                self.widget_tuples.append((widget, widgets_seen))
                widgets_seen += len(elem)
            else:
                layout = self.get_namefunc(elem, widgets_seen)
                self.widget_tuples.append((layout, widgets_seen))
                widgets_seen += 1
        
    def separate_celllist(self):
        """Return celllist separated into groups

        Result value is a list of single cells or lists of cells.  If
        a list of cell, each element in the list should be a simple
        cell.  Single cells are for cells that contain other cells.
        """
        result = []
        cur_section = [] # hold the current list of section cells
        for cell in self.celllist:
            if cell.isparent():
                if cur_section:
                    result.append(cur_section)
                    cur_section = []
                result.append(cell)
            else: cur_section.append(cell)
        if cur_section: result.append(cur_section)
        return result

    def makeLayout(self):
        """Return the appropriate layout for the orientation"""
        if self.orientation == 'wide': self.layout = QtGui.QHBoxLayout()
        elif self.orientation == 'tall': self.layout = QtGui.QVBoxLayout()
        else: assert False, "Bad orientation "+self.orientation
        self.layout.setSpacing(0)
        self.setLayout(self.layout)

    def redraw(self, cell):
        """Reread and redraw the widget for the given cell"""
        index = self.celllist.index(cell)
        print "List redraw of", cell.abs_ref
        for wt_index, (l, i) in enumerate(self.widget_tuples):
            if index == i: break
        else: assert False, "Index %d doesn't have it's own widget" % (index,)
        assert isinstance(l, QtGui.QLayout), "Expected NameFunc layout"

        # Remove and destroy old widget, and add new one
        self.layout.removeItem(l)
        l.hide()
        new_nf_layout = self.get_namefunc(cell, index)
        self.widget_tuples[wt_index] = (new_nf_layout, index)
        self.layout.insertLayout(index, new_nf_layout)


class Supervisor(QtCore.QObject):
    """GUI Controller/Supervisor class

    There should be one of this class per GUI instance.  It binds
    global signals and keeps track of gui-wide variables like the
    active cell.
    """
    def __init__(self, valchanged_callback):
        """Initialize with function to call when a value has changed"""
        QtCore.QObject.__init__(self)
        self.active_cell = self.root_gc = None
        self.valchanged_callback = valchanged_callback

    def set_root(self, root_gc, root_redraw_callback):
        """Set the root guicell, and what to do when root has new widget"""
        self.root_gc = root_gc
        assert len(root_gc.abs_ref) == 0
        self.root_redraw_callback = root_redraw_callback
        if not self.active_cell: self.active_cell = root_gc
        self.bind_signals_recursive(root_gc)
        
    def bind_signals_recursive(self, gc):
        """Catch the editingFinished and valueChanged signals"""
        self.connect(gc, QtCore.SIGNAL('editingFinished'), self.editingFinished)
        # This must be queued below because we don't want to re-enter
        # emitting widgets, which will be destroyed
        self.connect(gc, QtCore.SIGNAL('valueChanged'), self.valueChanged,
                     QtCore.Qt.QueuedConnection)
        if gc.isparent() and not gc.istable():
            for sub_gc in gc.value:
                self.bind_signals_recursive(sub_gc)

    def flip_slot(self):
        """This is called when a flip signal is received"""
        if self.active_cell.isparent(): flip_cell = self.active_cell
        else: flip_cell = self.get_parent(self.active_cell)
        print "Flipping", flip_cell.abs_ref
        flip_cell.flip_orientation()
        # Now make sure container redraws
        if flip_cell.isroot(): 
            self.root_redraw_callback(self.root_gc.value_widget)
        else: self.get_parent(flip_cell).redraw(flip_cell)

    def editingFinished(self, comptype, ar):
        """This should get called when a GuiCell has been edited"""
        print 'Super edit finish:', comptype, ar
        self.active_cell = self.root_gc.get_ref(ar)

    def valueChanged(self, comptype, ar, text):
        """This should get called when a GuiCell value has changed"""
        print 'Super value change:', comptype, ar, text
        self.valchanged_callback(comptype, ar, text)
        
    def get_parent(self, cell):
        """Return the parent of the given guicell"""
        assert not cell.isroot()
        return self.root_gc.get_ref(cell.abs_ref.parent())

    def get_main_widget(self):
        return self.root_gc.value_widget
