# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

class LazyArray:
	"""Lazy ordered associative array
	
	A LazyArray is a map from an ordered list of keys to values. It's lazy in
	the sense that values are stored as thunks which are only run when that
	value is requested.
	
	A LazyArray can have multiple blank (objects.Blank) values it its keylist,
	but Blank cannot be used to look up an item.
	
	"""
	def __init__(self, length):
		"""Return an empty lazyarray with the given length"""
		self.keys = [Blank]*length
		self.values = [lambda: Blank]*length

	def set_key(self, index, key):
		"""Set the key at index to given key"""
		self.keys[index] = Blank
		assert key == Blank or key not in self.keys, (key, self.keys)
		self.keys[index] = key

	def set_keys(self, keylist):
		"""Set the keys with a list of keys"""
		assert len(self.keys) == len(self), keylist
		keylist_wo_Nones = [elem for elem in keylist
							if not isinstance(elem, ExBlank)]
		assert len(keylist_wo_Nones) == len(set(keylist_wo_Nones)), keylist
		self.keys = keylist
		
	def set_value(self, index, thunk):
		"""Set the value at index to the given thunk.  Memoize value."""
		# below the first element indicates whether the thunk has been called
		# yet. The second element in the pair is the computed value.
		memo_pair = [False, None]
		def memoized_thunk():
			if not memo_pair[0]:
				memo_pair[1] = thunk()
				memo_pair[0] = True
			return memo_pair[1]
		self.values[index] = memoized_thunk

	def set_values_precomputed(self, valuelist):
		"""Set the values from a pre-computed list (no thunks)"""
		def make_thunk(val):
			def thunk(): return val
			return thunk
		assert len(valuelist) == len(self), valuelist
		self.values = [make_thunk(elem) for elem in valuelist]

	def set_items_precomputed(self, item_pair_list):
		"""Set values and keys from list like ((key1, val1), (key2, val2)..)"""
		keys = [key for key, val in item_pair_list]
		vals = [val for key, val in item_pair_list]
		self.set_keys(keys)
		self.set_values_precomputed(vals)

	def __getitem__(self, key):
		"""Return the value corresponding to key"""
		#assert isinstance(key, objects.ExObject), key
		if key is Blank: raise AttributeError(key)
		try: i = self.keys.index(key)
		except ValueError: raise AttributeError(key)
		return self.values[i]()

	def todict(self):
		"""Convert to dictionary, omitting Blank keys"""
		return dict([pair for pair in zip(self.keys, self.values)
					 if pair[0] != Blank])

	def clear_value(index):
		"""Remove the value at index"""
		self.values[index] = lambda: Blank

	def __len__(self): return len(self.keys)
	
	def totuple_recursive(self):
		"""Convert lazy array to tuple recursively.
		
		This is useful for testing purposes. Names are ignored---only values are
		included in the tuple.  Also, this unboxes any ExObjects.
		"""
		def helper(val):
			if isinstance(val, LazyArray): return val.totuple_recursive()
			else: return val
		return tuple(helper(val().obj) for val in self.values)

	def __str__(self): return "LazyArray: "+str(self.topairs_recursive())
	
	def __call__(self, key): return self[key]
	
	def __eq__(self, other):
		"""True if self and other are equal by value"""
		if not isinstance(other, LazyArray): return False
		if len(other) != len(self): return False
		if self.keys != other.keys: return False
		for i in range(len(self)):
			if self.values[i]() != other.values[i]():
				return False
		return True
	def __ne__(self, other): return not (self == other)
	
	def topairs(self):
		"""Return tuple of (name, value) pairs.  This runs all thunks"""
		return tuple((name, value())
					 for name, value in zip(self.keys, self.values))
	
	def topairs_recursive(self):
		"""Like topairs, but recur if value is another lazy array"""
		return tuple((name, value().topairs_recursive()
							if isinstance(value(), LazyArray) else value())
					 for name, value in zip(self.keys, self.values))

	def copy(self):
		"""Return a shallow copy"""
		new = LazyArray(len(self))
		new.keys = self.keys[:]
		new.values = self.values[:]
		return new

	def get_thunk(self, index):
		"""Return the value-producing thunk at the given index"""
		return self.values[index]

	def get_key(self, index):
		"""Return the name at the given index"""
		return self.keys[index]

	def get_keys(self):
		"""Return list of all keys in order"""
		return self.keys

	def deref_absref(self, abs_ref):
		"""Return the object at the given absolute reference"""
		if len(abs_ref) == 0: return self
		elem = self.get_thunk(abs_ref.car())()
		if len(abs_ref) == 1: return elem
		assert isinstance(elem, ExArray), (elem, type(elem), abs_ref.t)
		return elem.obj.deref_absref(abs_ref.cdr())

	def deref_absref_key(self, abs_ref):
		"""Return the key of the given absolute reference"""
		elem = self.deref_absref(abs_ref.parent())
		assert isinstance(elem, LazyArray), (elem, type(elem), abs_ref.t)
		return elem.get_key(abs_ref.last())

	def distinct(self):
		"""Return LazyArray like self, but with no duplicate values."""
		values_computed = set()
		values = []
		for thunk in self.values:
			value = thunk()
			if value not in values_computed:
				values_computed.add(value)
				values.append(value)
		la = LazyArray(len(values))
		la.set_values_precomputed(values)
		return la

	def sort(self):
		"""Return LazyArray like self but sorted.  Preserves names"""
		ex_values = zip(self.keys, [thunk() for thunk in self.values])
		ex_values.sort(cmp=lambda x,y: cmp(x[1], y[1]))
		la = LazyArray(len(self))
		la.set_items_precomputed(ex_values)
		return la

	def sortOn(self, sorting_la):
		"""Like sort above, but sort according to sorting_la"""
		assert len(self) == len(sorting_la), (len(self), len(sorting_la))
		ex_triples = zip(self.keys, self.values,
						 [thunk() for thunk in sorting_la.values])
		ex_triples.sort(cmp=lambda x,y: cmp(x[2], y[2]))
		la = LazyArray(len(self))
		la.keys = [key for key, thunk, sortkey in ex_triples]
		la.values = [thunk for key, thunk, sortkey in ex_triples]
		return la
		
	def reverse(self):
		"""Return LazyArray like self but with names/values in opposite order"""
		la = LazyArray(len(self))
		la.keys = self.keys[::-1]
		la.values = self.values[::-1]
		return la

def scale_up(scalar, size):
	"""Convert scalar into an array of given size by repeating it"""
	la = LazyArray(size)
	for i in range(size): la.set_value(size, lambda: scalar)
	return la


class LazyArrayStack:
	"""Stack of Lazy Arrays for nested scopes
	
	This object contains a sequence of LazyArrays. When given a key, it can
	return the item in the first LA that has it.
	"""
	def __init__(self):
		"""Create an emtpy LazyArrayStack"""
		self._stack = ()

	def __getitem__(self, key):
		"""Search through the array stack, return value associated with key"""
		for la in self._stack:
			try: return la[key]
			except AttributeError: pass
		raise AttributeError(key)

	def __add__(self, lazy_array):
		"""Return new LazyArrayStack with extra LA on top"""
		new_las = LazyArrayStack()
		new_las._stack = (lazy_array,) + self._stack
		return new_las

	def get_root_la(self):
		"""Return the root lazy array in the stack
		
		This is a hack and needs to be updated---the idea is that the very
		bottom one is the default globals dictionary, so the second to bottom
		will be the actual root lazy array.
		
		"""
		assert len(self._stack) >= 2, self._stack
		return self._stack[-2]
		

# put at end to avoid circular importing problem
from objects import ExBlank, Blank
from objects import ExArray
