# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

"""This module defines the basic objects that Explicans manipulates"""

import types

types = ('num', 'string', 'bool', 'blank', 'ref', 'func', 'array', 'table')
scalars = ('num', 'string', 'bool', 'blank', 'ref')

class ExObject:
	"""Base class for all Explicans objects

	For method access, call the get_method method. It reads the self.attrs
	dictionary, which is a map from python strings to pairs (python method,
	number of arguments). If number of arguments is 0, the method lookup returns
	the results of the function instead of the function itself.
	
	"""
	def __init__(self, t, obj):
		"""Initialize object with type t and python equivalent object"""
		assert t in types, t
		self.t = t
		self.obj = obj
		self.relref_info = None # Set to RelRefInfo object
		self.attrs = {'blank?':(self.isblank, ())}

	def get_method(self, method_name):
		"""Return the method referenced by method_name"""
		if self.relref_info:
			result = self.relref_info.get_method(method_name)
			if result is not None: return result
		func, scalar_list = self.attrs[method_name]
		return func() if len(scalar_list) == 0 else ExFunc(func, scalar_list)

	def set_method(self, methodname, scalar_list, function):
		"""Add a method to the object.  See ExFunc for scalar_list def."""
		self.attrs[methodname] = (function, scalar_list)

	def set_relref(self, relref_info):
		"""Set RelRefInfo object, activate relative reference semantics"""
		self.relref_info = relref_info

	def isblank(self):
		"""Return ExTrue if object is blank, false otherwise"""
		if isinstance(self, ExBlank): return ExTrue
		return ExFalse

	def isscalar(self):
		"""Return ExTrue if object has scalar type"""
		return self.t in scalars

	def __str__(self): return "ExObj: %s, %s" % (self.t, self.obj)
	def __eq__(self, other):
		return (isinstance(other, ExObject) and
				self.t == other.t and self.obj == other.obj)

	def __hash__(self):
		"""Identify hash with type and object"""
		return hash((self.t, self.obj))
		
	def get_type(self): return self.t
	
	def __getitem__(self, key): return self.obj[key]


class ExString(ExObject):
	"""Explicans String class"""
	def __init__(self, string):
		"""Initialize with a python string"""
		ExObject.__init__(self, 'string', string)
		self.set_method('upper', (), lambda: ExString(self.obj.upper()))
		self.set_method('lower', (), lambda: ExString(self.obj.lower()))

	def __hash__(self): return hash(self.obj)
	def __cmp__(self, other): return cmp(self.obj, other.obj)
	
class ExNum(ExObject):
	"""Explicans number (floating point) class"""
	def __init__(self, num):
		"""Initialize with a number"""
		ExObject.__init__(self, 'num', num)

	def __hash__(self): return hash(self.obj)
	def __cmp__(self, other): return cmp(self.obj, other.obj)

class ExArray(ExObject):
	"""Explicans Lazy Array class"""
	def __init__(self, la):
		"""Initialize with a lazy array"""
		ExObject.__init__(self, 'array', la)
		self.scalar_list = (True,)*len(self.obj)
		self.set_method('len', (), lambda: ExNum(self.len()))
		self.set_method('distinct', (), self.distinct)
		self.set_method('sort', (), self.sort)
		self.set_method('sortOn', (False,), self.sortOn)
		self.set_method('reverse', (), self.reverse)

	def len(self): return len(self.obj)
	
	def distinct(self):
		"""Return ExArray containing the distinct items of self"""
		return ExArray(self.obj.distinct())

	def sort(self):
		"""Return ExArray with same elements of self in sorted order"""
		return ExArray(self.obj.sort())

	def sortOn(self, sorting_exa):
		"""Return ExArray version of self, sorted by sorting_exa"""
		assert isinstance(sorting_exa, ExArray)
		return ExArray(self.obj.sortOn(sorting_exa.obj))

	def reverse(self):
		"""Return ExArray like self but with elements in reverse order"""
		return ExArray(self.obj.reverse())


class ExFunc(ExObject):
	"""Explicans function object"""
	def __init__(self, func, scalar_list):
		"""Initialize with a function object and list of argument types

		argtypes should be a list signifying whether each argument should be
		scalar. For instance [True, False, True] means that the function takes
		three arguments. The first and third are scalars but the second can be
		an array/function/whatever.
		
		"""
		ExObject.__init__(self, 'func', func)
		for elem in scalar_list:
			assert isinstance(elem, type(False)), scalar_list
		self.scalar_list = scalar_list

class ExBool(ExObject):
	"""Explicans boolean object"""
	def __init__(self, py_boolean):
		"""Initialize with True or False"""
		assert py_boolean is True or py_boolean is False
		ExObject.__init__(self, 'bool', py_boolean)
ExTrue = ExBool(True)
ExFalse = ExBool(False)

class ExBlank(ExObject):
	"""Explicans blank (None) object"""
	def __init__(self):
		ExObject.__init__(self, 'blank', None)
Blank = ExBlank()

import lazyarray # put at end to avoid circular importing problem
