# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

import cStringIO, code
from output import antlr3
from output import ExprLexer, ExprParser # Load antlr-generated code

class ParsingException(Exception): pass # generic parsing exception

def testInteractiveDebug(local_dict):
	"""If we need an interactive debugger, can start one with this
	
	Usually run with locals(), for instance:
		testInteractiveDebug(locals())
	"""
	code.interact(banner = "Welcome to debugging console",
				  readfunc = raw_input, local = local_dict)

def parse_string(s):
	"""Return the syntax tree of formula s in tuple form
	
	This function uses antlr-generated code to parse an expression. If the
	expression does not parse, return None.
	
	The output will be similar to lisp S-expressions. For instance,
	parse_string('3*b+2') goes to
	('+', ('*', ('NUM', '3'), ('NAME', 'b')), ('NUM', '2')).
	See the test cases for more examples.
	"""
	tree = _get_common_tree(s)
	return _descend_tree(tree) if tree else None

def _get_common_tree(expr_string):
	"""Return the CommonTree object created by antlr3, or None if error"""
	stringio = cStringIO.StringIO(expr_string)
	char_stream = antlr3.ANTLRInputStream(stringio)
	lexer = ExprLexer.ExprLexer(char_stream)
	tokens = antlr3.CommonTokenStream(lexer)
	parser = ExprParser.ExprParser(tokens)
	#expr = parser.expr_all()
	#print "Tokens: "+','.join(repr(t.text) for t in tokens.tokens)
	#print "@", expr_string
	#testInteractiveDebug(locals())
	expr = _get_expr(parser, expr_string)
	return expr.tree if expr else None

def _get_expr(antlr_parser, expr_string):
	"""Get expression parse, or None if parsing problem"""
	try: expr = antlr_parser.expr_all() # expr_all is the rule to match
	except antlr3.RecognitionException as e:
		return None
	except antlr3.NoViableAltException as e:
		return None
	return expr

def _descend_tree(common_tree):
	"""Descend an antlr CommonTree recursively and convert to simple tuple"""
	children = common_tree.getChildren()
	if not children: return common_tree.getText()
	child_tuples = [_descend_tree(child_tree) for child_tree in children]
	parent_text = common_tree.getText()

	# Now do any fixing or overriding of parser output
	if parent_text == 'NAME': return (parent_text, ''.join(child_tuples))
	child_tuples = [x for x in child_tuples if x != ' '] # ignore whitespace
	if parent_text == 'STRING':
		assert len(child_tuples) == 1, child_tuples
		return (parent_text, unquote(child_tuples[0]))
	return (parent_text,) + tuple(child_tuples)	

def unquote(s):
	"""Unquote the quoted string s, processing escape sequences"""
	assert s[0] == s[-1] == '"', s
	s = s[1:-1]
	start_index = 0
	while True:
		i = s.find('\\', start_index)
		if i == -1: return s
		s = s[:i]+s[i+1:]
		start_index = i+1 # skip the quoted character
