# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

from PyQt4 import QtGui

"""Make QSS stylesheets"""

border_color = 'rgb(80, 80, 80)'
background_color = border_color
name_color = 'rgb(206, 206, 206)'
name_qcolor = QtGui.QColor(QtGui.qRgb(206, 206, 206))
func_color = 'rgb(214, 235, 255)'
func_qcolor = QtGui.QColor(QtGui.qRgb(214, 234, 255))
value_color = 'white'

bevel_size = '2ex'
edge_padding = '.5ex'

background = 'background-color: %s;' % (background_color,)

def get_style(comptype, orientation, desc_set):
    """Return the style sheet appropriate for the given widget

    comptype should either be "name", "func", or "value"

    orientation should either be "wide" or "tall"

    desc_set should be a set of strings.  Here are the key values:
    top, bottom, left, right - means the widget is on the top,
    bottom, left, or right of the table.

    We want there to be a 1px border between each box, so in
    general each only draws a bottom and right border.  However,
    at the top and left edges, do draw a border.

    Additionally, boxs in a corner with a bevel need more padding
    to pull that off.
    """
    if comptype == 'name': main_color = name_color
    elif comptype == 'func': main_color = func_color
    elif comptype == 'value': main_color = value_color
    else: assert False, ('Unknown comptype ' + comptype)
    statements = ['margin: 0px;',
                  'border-style: solid;',
                  'border-width: 1px;',
                  'background-color: %s;' % (main_color,)]

    # Set extra padding around edge and top/left borders
    for edge in ('top', 'left'):
        if edge in desc_set:
            statements.append('padding-%s: %s;' % (edge, edge_padding,))
            statements.append('border-%s-color: %s;' % (edge, border_color,))
        else:
            statements.append('padding-%s: 0px;' % (edge,))
            statements.append('border-%s-color: %s;' % (edge, main_color,))
    for edge in ('bottom', 'right'):
        if edge in desc_set:
            statements.append('padding-%s: %s;' % (edge, edge_padding,))
        else: statements.append('padding-%s: 0px;' % (edge,))

    # Add bottom and/or right borders where necessary
    if orientation == 'tall':
        bottom_border, right_border = border_color, main_color
    else: bottom_border, right_border = main_color, border_color
    statements.append('border-bottom-color: %s;' % (bottom_border,))
    statements.append('border-right-color: %s;' % (right_border,))

    # Add beveling at corners
    for top_bottom in ('top', 'bottom'):
        for right_left in ('right', 'left'):
            if top_bottom in desc_set and right_left in desc_set:
                statements.append('border-%s-%s-radius: %s;' %
                                  (top_bottom, right_left, bevel_size))
            else: statements.append('border-%s-%s-radius: 0px;' %
                                    (top_bottom, right_left))

    return '\n'.join(statements)

def get_table_value_style(desc_set):
    """Return the stylesheet for a value widget in a 2D table
    
    desc_set may contain 'top', 'bottom', 'left', and 'right' if the value is on
    a corner/edge.  Otherwise the widget will be styled for the middle.
    """
    statements = ['margin: 0px;',
                  'border-style: solid;',
                  'border-width: 1px;',
                  'background-color: %s;' % (value_color,)]
    minor_border_color = name_color # used between cells
    major_border_color = border_color # used at top and left
    for edge in ('top', 'left'):
        if edge in desc_set:
            statements.append('padding-%s: %s;' % (edge, edge_padding))
            statements.append('border-%s-color: %s;'
                              % (edge, major_border_color))
        else:
            statements.append('padding-%s: 0px;' % (edge,))
            statements.append('border-%s-color: %s;' % (edge, value_color))

    if 'bottom' in desc_set and 'right' in desc_set:
        statements.append('border-bottom-right-radius: %s;' % (bevel_size,))
        statements.append('border-top-left-radius: 0px;')
    else:
        statements.append('border-bottom-color: %s;' % (minor_border_color,))
        statements.append('border-right-color: %s;' % (minor_border_color,))

    return '\n'.join(statements)
