# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.


"""PyQt4 port of the layouts/basiclayout example from Qt v4.x"""

import sys, types
from PyQt4 import QtCore, QtGui
import stylesheets

FIXED_WIDTH = 100

class ListSection(QtGui.QWidget):
    """A one dimensional table of simple cells"""
    coord_dict = {0: 'name', 1: 'func', 2: 'value'}
    comp_dict = {'name': 0, 'func': 1, 'value': 2}
    def __init__(self, orientation, length, begin = True, end = True):
        """Initialize with 'wide' or 'tall' and the number of elements

        Optionally, specify begin and/or end as False to indicate that
        this table section is not the beginning/end of the overall
        table.
        """
        QtGui.QWidget.__init__(self)
        self.setSize(orientation, length)
        self.begin, self.end = begin, end

        self.set_widget_dict()
        self.add_widget_styles()
        self.add_widgets_to_layout()
        self.bind_signals()
        self.setLayout(self.layout)        

    def setSize(self, orientation, length):
        """Set the orientation and size attributes"""
        if orientation == 'wide': self.rows, self.cols = 3, length
        elif orientation == 'tall': self.rows, self.cols = length, 3
        else: assert False, ('Invalid orientation '+orientation)
        self.orientation, self.length = orientation, length

    def get_widget(self, comptype, pos):
        """Return the widget at position pos and func/name/value"""
        if self.orientation == 'tall':
            return self.widget_dict[(pos, self.comp_dict[comptype])]
        else: return self.widget_dict[(self.comp_dict[comptype], pos)]

    def get_pos(self, row, col):
        """Return (comptype, position) from row and col coordinates"""
        if self.orientation == 'tall': return (self.coord_dict[col], row)
        else: return (self.coord_dict[row], col)

    def set_widget_dict(self):
        """Sets initial dictionary mapping (row, col) coords to widgets"""
        self.widget_dict = {}
        for i in range(self.rows):
            for j in range(self.cols):
                comptype = self.get_pos(i,j)[0]
                if comptype == 'value': w = QtGui.QLabel()
                else: w = QtGui.QLineEdit()
                w.setFixedWidth(FIXED_WIDTH)
                self.widget_dict[(i,j)] = w

    def add_widget_styles(self):
        """Process widget dictionary and add style sheets"""
        desc_sets = {}
        def add_to_desc_sets(i, j, desc):
            """Add description to desc_set at position i, j in desc_sets"""
            try: desc_set = desc_sets[(i,j)]
            except KeyError: desc_sets[(i,j)] = set((desc,))
            else: desc_set.add(desc)

        # Mark sides that apply even in middle of block
        if self.orientation == 'tall':
            for i in range(self.rows):
                add_to_desc_sets(i, 0, 'left')
                add_to_desc_sets(i, self.cols-1, 'right')
        else:
            for j in range(self.cols):
                add_to_desc_sets(0, j, 'top')
                add_to_desc_sets(self.rows-1, j, 'bottom')

        # Mark beginning and end of tables
        if self.begin:
            if self.orientation == 'tall':
                for j in range(self.cols):
                    add_to_desc_sets(0, j, 'top')
            else:
                for i in range(self.rows):
                    add_to_desc_sets(i, 0, 'left')
        if self.end:
            if self.orientation == 'tall':
                for j in range(self.cols):
                    add_to_desc_sets(self.rows-1, j, 'bottom')
            else:
                for i in range(self.rows):
                    add_to_desc_sets(i, self.cols-1, 'right')

        # Finally make the style sheets and apply them
        for i in range(self.rows):
            for j in range(self.cols):
                if (i,j) in desc_sets: desc_set = desc_sets[i,j]
                else: desc_set = set()
                sheet = stylesheets.get_style(self.get_pos(i,j)[0],
                                              self.orientation, desc_set)
                self.widget_dict[(i,j)].setStyleSheet(sheet)

    def add_widgets_to_layout(self):
        """Create layout and add widgets in widget dictionary to it"""
        self.layout = QtGui.QGridLayout()
        self.layout.setSpacing(0)
        #Align statement below may make sense but doesn't look right
        #align = QtCore.Qt.AlignTop | QtCore.Qt.AlignLeft

        for i in range(self.rows):
            for j in range(self.cols):
                widget = self.widget_dict[(i,j)]
                self.layout.addWidget(widget, i, j)

    def bind_signals(self):
        """Connect the signals from widget edits to editing finished"""
        def helper(i,j): #use lexical scoping to call with coords
            def editing_finished():
                """This is called when a widget is edited"""
                comptype, position = self.get_pos(i,j)
                print "TS: editing finished:", comptype, position
                self.emit(QtCore.SIGNAL('editingFinished'),
                          comptype, position, self.widget_dict[(i,j)].text())
            return editing_finished

        for i in range(self.rows):
            for j in range(self.cols):
                self.connect(self.widget_dict[(i,j)],
                             QtCore.SIGNAL('editingFinished()'),
                             helper(i,j))

    def setText(self, triple_list):
        """Set the text in the widgets

        Format of argument is a list of (name, func, value) strings
        """
        for index, triple in enumerate(triple_list):
            for s, comptype in zip(triple, ('name', 'func', 'value')):
                self.get_widget(comptype, index).setText(s)

class NameFuncLayoutBase:
    """Base class for NameFuncHLayout and NameFuncVLayout"""
    align = QtCore.Qt.AlignTop | QtCore.Qt.AlignLeft
    def set_elems(self, name, func, value_widget):
        self.setSpacing(0)
        self.nf = NameFunc(self.orientation)
        self.nf.setText(name, func)
        self.addWidget(self.nf, 0, self.align)

        self.value_widget = value_widget
        self.addWidget(value_widget, 0, self.align)
        self.addStretch(1)
    def hide(self):
        """Hide all the subwidgets"""
        self.nf.hide()
        self.value_widget.hide()

class NameFuncHLayout(QtGui.QHBoxLayout, NameFuncLayoutBase):
    def __init__(self, name, func, value_widget):
        QtGui.QHBoxLayout.__init__(self)
        self.orientation = 'tall'
        NameFuncLayoutBase.set_elems(self, name, func, value_widget)
class NameFuncVLayout(QtGui.QVBoxLayout, NameFuncLayoutBase):
    def __init__(self, name, func, value_widget):
        QtGui.QVBoxLayout.__init__(self)
        self.orientation = 'wide'
        NameFuncLayoutBase.set_elems(self, name, func, value_widget)

class NameFunc(QtGui.QWidget):
    """Hold an isolated name and function for when value is array"""
    def __init__(self, orientation):
        """Initialize with 'wide' or 'tall' as the orientation

        This widget is actually wider when you specify 'tall', but the
        idea is the wide widget will match 'wide' tables and
        vice-versa.
        """
        QtGui.QWidget.__init__(self)
        self.orientation = orientation
        self.makeLayout()
        self.set_namefuncwidgets()
        self.bind_signals()

    def makeLayout(self):
        """Return the appropriate layout for the orientation"""
        if self.orientation == 'wide': self.layout = QtGui.QVBoxLayout()
        elif self.orientation == 'tall': self.layout = QtGui.QHBoxLayout()
        else: assert False, "Bad orientation "+self.orientation
        self.layout.setSpacing(0)
        self.setLayout(self.layout)

    def set_namefuncwidgets(self):
        """Make the name and function widgets and set them"""
        self.name_widget = QtGui.QLineEdit()
        self.func_widget = QtGui.QLineEdit()

        name_style = stylesheets.get_style('name', self.orientation, set())
        func_style = stylesheets.get_style('func', self.orientation, set())
        self.name_widget.setStyleSheet(name_style)
        self.func_widget.setStyleSheet(func_style)
        self.name_widget.setFixedWidth(FIXED_WIDTH)
        self.func_widget.setFixedWidth(FIXED_WIDTH)

        self.layout.addWidget(self.name_widget)
        self.layout.addWidget(self.func_widget)

    def setText(self, name_text, func_text):
        """Set the name and function text"""
        self.name_widget.setText(name_text)
        self.func_widget.setText(func_text)

    def bind_signals(self):
        """Emit an editingFinished signal when subwidget modified"""
        def helper(comptype):
            def editing_finished():
                """This is called when the name or func widget is changed"""
                print 'NF: editing finished:', comptype
                if comptype == 'name': text = self.name_widget.text()
                else: text = self.func_widget.text()
                self.emit(QtCore.SIGNAL('editingFinished'), comptype, 0, text)
            return editing_finished

        self.connect(self.name_widget, QtCore.SIGNAL('editingFinished()'),
                     helper('name'))
        self.connect(self.func_widget, QtCore.SIGNAL('editingFinished()'),
                     helper('func'))


class Table2D(QtGui.QWidget):
    """A two dimensional table
    
    This is just based on a grid of smaller widgets using QGridLayout. I tried
    at first using QTableWidget but I wasn't sure how to get the headers to come
    out right.
    
    This widget sends out signal valueChanged when the text in one of the
    widgets is changed.  The arguments are:

    comptype - either "func" or "name"
    axis - right now this is either 'C' or 'R', depending on whether one of the
    names/formula for the columns or rows was edited.
    index - The row or column that was edited. This is 0 if the name/formula of
    the rows/columns themselves were edited.
    text - the new text value
    """
    def __init__(self, height, width):
        """Initialize with number of rows and columns"""
        QtGui.QWidget.__init__(self)
        self.height, self.width = height, width
        self.init_grid()
        self.setLayout(self.grid)
    
    def make_vc_signal_handler(self, comptype, axis, index):
        """Return the function to be called when some widget's value changes"""
        def vc_signal_handler():
            d = self.name_widgets if comptype == 'name' else self.form_widgets
            changed_widget = d[(axis, index)]
            self.emit(QtCore.SIGNAL('editingFinished'),
                      comptype, axis, index, changed_widget.text())
        return vc_signal_handler

    def init_grid(self):
        """Create the grid layout and add the right widgets"""
        self.grid = QtGui.QGridLayout()
        self.grid.setSpacing(0)
        self.value_style = stylesheets.get_style('value', 'wide', set())

        self.name_widgets = {} # keys are (axis, index), values are widgets
        self.form_widgets = {} # keys are (axis, index), values are widgets
        self.value_widgets = {} # keys are (axis, index), values are widgets
        self.add_column_desc()
        self.add_row_desc()
        self.add_individual_row_names()
        self.add_individual_col_desc()
        self.add_value_widgets()

    def add_row_desc(self):
        """Add the widgets for the row names and formulas"""
        # First add the name widget
        row_name_w = QtGui.QLineEdit()
        name_style = stylesheets.get_style('name', 'wide', set(['top', 'left']))
        row_name_w.setStyleSheet(name_style)
        self.grid.addWidget(row_name_w, 1, 0)
        self.connect(row_name_w, QtCore.SIGNAL('editingFinished()'),
                     self.make_vc_signal_handler('name', 'R', 0))
        self.name_widgets[('R', 0)] = row_name_w
        
        # Now do the same for the formula widget
        formula_style = stylesheets.get_style('func', 'wide', set())
        row_formula_w = QtGui.QLineEdit()
        row_formula_w.setStyleSheet(formula_style)
        self.grid.addWidget(row_formula_w, 2, 0)
        self.connect(row_formula_w, QtCore.SIGNAL('editingFinished()'),
                     self.make_vc_signal_handler('func', 'R', 0))
        self.form_widgets[('R', 0)] = row_formula_w

    def add_column_desc(self):
        """Add the widgets for the column names and formulas"""
        # First add the name widget
        col_name_w = QtGui.QLineEdit()
        name_style = stylesheets.get_style('name', 'tall', set())
        col_name_w.setStyleSheet(name_style)        
        self.grid.addWidget(col_name_w, 0, 1)
        self.connect(col_name_w, QtCore.SIGNAL('editingFinished()'),
                     self.make_vc_signal_handler('name', 'C', 0))
        self.name_widgets[('C', 0)] = col_name_w

        # Now do the same for the formula widget        
        col_formula_w = QtGui.QLineEdit()
        formula_style = stylesheets.get_style('func', 'tall', set())
        col_formula_w.setStyleSheet(formula_style)
        self.grid.addWidget(col_formula_w, 0, 2)
        self.connect(col_formula_w, QtCore.SIGNAL('editingFinished()'),
                     self.make_vc_signal_handler('func', 'C', 0))
        self.form_widgets[('C', 0)] = col_formula_w

    def add_individual_row_names(self):
        style = stylesheets.get_style('name', 'tall', set())
        last_style = stylesheets.get_style('name', 'tall',
                                           set(['bottom', 'left']))
        for i in range(self.height):
            row_name_w = QtGui.QLineEdit()
            row_name_w.setStyleSheet(style
                                     if i < self.height-1 else last_style)
            self.grid.addWidget(row_name_w, i+3, 0)
            self.connect(row_name_w, QtCore.SIGNAL('editingFinished()'),
                         self.make_vc_signal_handler('name', 'R', i+1))
            self.name_widgets[('R', i+1)] = row_name_w

    def add_individual_col_desc(self):
        """Add the widgets for each column's name and description"""
        name_style = stylesheets.get_style('name', 'wide', set(['top']))
        lastname_style = stylesheets.get_style('name', 'wide',
                                               set(['top', 'right']))
        form_style = stylesheets.get_style('func', 'wide', set())
        for i in range(self.width):
            name_w = QtGui.QLineEdit()
            name_w.setStyleSheet(name_style
                                 if i < self.width-1 else lastname_style)
            self.grid.addWidget(name_w, 1, i+1)
            self.connect(name_w, QtCore.SIGNAL('editingFinished()'),
                         self.make_vc_signal_handler('name', 'C', i+1))
            self.name_widgets[('C', i+1)] = name_w
            
            form_w = QtGui.QLineEdit()
            form_w.setStyleSheet(form_style)
            self.grid.addWidget(form_w, 2, i+1)
            self.connect(form_w, QtCore.SIGNAL('editingFinished()'),
                         self.make_vc_signal_handler('func', 'C', i+1))
            self.form_widgets[('C', i+1)] = form_w

    def add_value_widgets(self):
        for i in range(self.height):
            for j in range(self.width):
                desc_set = set()
                if i == 0: desc_set.add('top')
                if i == self.height-1: desc_set.add('bottom')
                if j == 0: desc_set.add('left')
                if j == self.width-1: desc_set.add('right')
                value_w = QtGui.QLabel()
                style = stylesheets.get_table_value_style(desc_set)
                value_w.setStyleSheet(style)
                self.grid.addWidget(value_w, i+3, j+1)
                self.value_widgets[(i,j)] = value_w

    def set_value_text(self, i, j, val_string):
        """Set the text the value widget at i,j"""
        self.value_widgets[(i,j)].setText(val_string)

    def set_rowname_text(self, i, val_string):
        """Set the name of row i to val_string"""
        self.name_widgets[('R', i+1)].setText(val_string)
    
    def set_colname_text(self, j, val_string):
        """Set the name of column j to val_string"""
        self.name_widgets[('C', j+1)].setText(val_string)

    def set_axis_names(self, row_axis_str, col_axis_str):
        self.name_widgets[('R', 0)].setText(row_axis_str)
        self.name_widgets[('C', 0)].setText(col_axis_str)

    def set_axis_formulas(self, row_formula_text, col_formula_text):
        """Set the row axis formula text"""
        self.form_widgets[('R', 0)].setText(row_formula_text)
        self.form_widgets[('C', 0)].setText(col_formula_text)

    def set_colform_text(self, j, formula_text):
        """Set the column formula text for column j"""
        self.form_widgets[('C', j+1)].setText(formula_text)
    