;;;; "opticompute.scm" Optical stack and parameter extraction. -*-scheme-*-
;;; Copyright (C) 2003, 2004, 2006 Aubrey Jaffer

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

;; http://swiss.csail.mit.edu/~jaffer/FreeSnell

(require 'array)
(require 'subarray)
(require-if 'compiling 'printf)		; for DESCRIBE-STACK

(if (not (defined? real-ln))
    (define real-ln $log))

(define (avg2 x y) (/ (+ x y) 2))

(define (parameter-name tok)
  (case tok
    ((T_s) 's-transmission)
    ((R_s) 's-reflection)
    ((B_s) 's-reflection-back)
    ((A_s) 's-absorption)
    ((L_s) 's-absorption-back)
    ((T_p) 'p-transmission)
    ((R_p) 'p-reflection)
    ((B_p) 'p-reflection-back)
    ((A_p) 'p-absorption)
    ((L_p) 'p-absorption-back)
    ((T)   'tranmission)
    ((R)   'reflection)
    ((B)   'reflection-back)
    ((A)   'absorption)
    ((L)   'absorption-back)
    (else (slib:error 'parameter-name 'unknown 'token tok))))

(define (extract-parameter tok spol ppol)
  (letrec ((exsym
	    (lambda (tk)
	      (case tk
		((1) 1)
		((T_s) (car spol))
		((R_s) (cadr spol))
		((B_s) (caddr spol))
		((A_s) (- 1 (car spol) (cadr spol)))
		((L_s) (- 1 (car spol) (caddr spol)))
		((T_p) (car ppol))
		((R_p) (cadr ppol))
		((B_p) (caddr ppol))
		((A_p) (- 1 (car ppol) (cadr ppol)))
		((L_p) (- 1 (car ppol) (caddr ppol)))
		((T)   (avg2 (car spol) (car ppol)))
		((R)   (avg2 (cadr spol) (cadr ppol)))
		((B)   (avg2 (caddr spol) (caddr ppol)))
		((A)   (- 1 (avg2 (car spol) (car ppol))
			  (avg2 (cadr spol) (cadr ppol))))
		((L)   (- 1 (avg2 (car spol) (car ppol))
			  (avg2 (caddr spol) (caddr ppol))))
		(else (slib:error 'extract-parameter 'unknown 'token tk)))))
	   (evl (lambda (tk)
		  (cond ((number? tk) tk)
			((symbol? tk) (exsym tk))
			((pair? tk)
			 (case (car tk)
			   ((*) (apply * (map evl (cdr tk))))
			   ((+) (apply + (map evl (cdr tk))))
			   ((-) (apply - (map evl (cdr tk))))
			   ((/) (apply / (map evl (cdr tk))))
			   ((ln) (real-ln (evl (cadr tk))))
			   ((log_10) (real-log10 (evl (cadr tk))))
			   ((average)
			    (/ (apply + (map evl (cdr tk)))
			       (length (cdr tk))))
			   (else (slib:error 'extract-parameter 'unknown 'operator (car tk)))))
			(else (slib:error 'extract-parameter 'unknown 'expression tk))))))
    (evl tok)))

(define (extract-refractive w tok z)
  (define substrate (list '(1 0) (list z 0)))
  (case tok
    ((n real real-part) (real-part z))
    ((k imag imag-part ec extinction-coefficient) (imag-part z))
    ((k/n) (/ (imag-part z) (real-part z)))
    ((r0 reflect reflectance)
     (let ((spol (combine-layers 0 (- w) substrate))
	   (ppol (combine-layers 0 w     substrate)))
       (avg2 (cadr spol) (cadr ppol))))
    (else (slib:warn 'extract-refractive 'unknown 'token tok))))

(define (compute/wavelength wmin wmax samples incidents layers)
  (define data (make-array (A:floR64b)
			   samples
			   (+ 1 (count-plots incidents))))
  (define (average/incident angles w layers)
    (define cang (length angles))
    (define vals (combine-layers (car angles) w layers))
    (cond ((eqv? 1 cang) vals)
	  (else (for-each
		 (lambda (angle)
		   (set! vals (map + vals (combine-layers angle w layers))))
		 (cdr angles))
		(map (lambda (x) (/ x cang)) vals))))
  (define (iterate sdx w)
    (define idx 0)
    (array-set! data w sdx 0)
    (for-each
     (lambda (incident)
       (let ((angs (car incident)))
	 (define spol (average/incident angs (- w) layers))
	 (define ppol (average/incident angs w layers))
	 (for-each
	  (lambda (tok)
	    (set! idx (+ 1 idx))
	    (array-set! data (extract-parameter tok spol ppol) sdx idx))
	  (cdr incident))))
     incidents))
  (let ((wwmin (min wmax wmin)))
    (set! wmax (max wmax wmin))
    (set! wmin wwmin))
  (if (< 1.5 (/ wmax wmin) 10000)
      (let ((winc (real-expt (/ wmax wmin) (/ (+ -1 samples)))))
	(do ((cnt 0 (+ 1 cnt))
	     (w wmin (* w winc)))
	    ((>= cnt samples))
	  (iterate cnt w)))
      (let ((winc (/ (- wmax wmin) (+ -1 samples))))
	(do ((cnt 0 (+ 1 cnt))
	     (w wmin (+ w winc)))
	    ((>= cnt samples))
	  (iterate cnt w))))
  data)

(define (compute-IRs/wavelength wmin wmax samples IRs)
  (define data (make-array (A:floR64b)
			   samples
			   (+ 1 (count-plots IRs))))
  (define (iterate sdx w)
    (define idx 0)
    (array-set! data w sdx 0)
    (for-each
     (lambda (IR)
       (define z (if (number? (car IR)) (car IR) ((car IR) w)))
       (for-each
	(lambda (tok)
	  (set! idx (+ 1 idx))
	  (array-set! data (extract-refractive w tok z) sdx idx))
	(cdr IR)))
     IRs))
  (let ((wwmin (min wmax wmin)))
    (set! wmax (max wmax wmin))
    (set! wmin wwmin))
  (if (< 1.5 (/ wmax wmin) 10000)
      (let ((winc (real-expt (/ wmax wmin) (/ (+ -1 samples)))))
	(do ((cnt (+ -1 samples) (+ -1 cnt))
	     (w wmin (* winc w)))
	    ((negative? cnt))
	  (iterate cnt w)))
      (let ((winc (/ (- wmax wmin) (+ -1 samples))))
	(do ((cnt (+ -1 samples) (+ -1 cnt))
	     (w wmin (+ winc w)))
	    ((negative? cnt))
	  (iterate cnt w))))
  data)

(define (compute/angle thmin thmax rad->deg samples wvspecs layers)
  (define data (make-array (A:floR64b) samples (+ 1 (count-plots wvspecs))))
  (define thinc (/ (- thmax thmin) (+ -1 samples)))
  (do ((cnt (+ -1 samples) (+ -1 cnt))
       (th thmin (+ thinc th)))
      ((negative? cnt))
    (array-set! data (rad->deg th) cnt 0)
    (let ((idx 0))
      (for-each
       (lambda (wvspec)
	 (define spol (combine-layers th (- (car wvspec)) layers))
	 (define ppol (combine-layers th (car wvspec) layers))
	 (for-each
	  (lambda (tok)
	    (set! idx (+ 1 idx))
	    (array-set! data (extract-parameter tok spol ppol) cnt idx))
	  (cdr wvspec)))
       wvspecs)))
  data)

(define (optical-stack . args)
  (define layers '())
  (define wnom #f)
  (define (do-arg arg)
    (cond ((not (and (list? arg) (pair? arg)))
	   (slib:warn 'optical-stack 'mystery arg))
	  (else (case (car arg)
		  ((repeat)
		   (do ((cnt (+ -1 (cadr arg)) (+ -1 cnt)))
		       ((negative? cnt))
		     (for-each do-arg (cddr arg))))
		  ((nominal)
		   (set! wnom (cadr arg)))
		  ((layer)
		   (let ((thick (caddr arg)))
		     (if (not (null? (cdddr arg)))
			 (set! wnom (cadddr arg)))
		     (if (null? layers) (set! layers '((1 0))))
		     (set! layers
			   (cons
			    (if (> thick 1/50)
				(list (cadr arg)
				      (/ (* wnom thick)
					 (real-part
					  (if (number? (cadr arg))
					      (cadr arg)
					      ((cadr arg) wnom))))
				      wnom)
				(if wnom
				    (list (cadr arg) thick wnom)
				    (cdr arg)))
			    layers))))
		  ((substrate)
		   (set! layers (cons (list (cadr arg) 0) layers)))
		  (else
		   (slib:warn 'optical-stack 'unknown arg))))))
  (for-each do-arg args)
  (merge-layers (if (zero? (cadar layers))
		    layers
		    (cons '(1 0) layers))))

;;;also reverses
(define (merge-layers stack)
  (define (loop layers layer stack)
    (cond ((null? layers) (cons layer stack))
	  ((and (eqv? (caar layers) (car layer))
		(eqv? (caddar layers) (caddr layer))
		(eqv? (negative? (cadar layers)) (negative? (cadr layer))))
	   (loop (cdr layers)
		 (list (car layer)
		       (+ (cadar layers) (cadr layer))
		       (caddr layer))
		 stack))
	  ((and (zero? (cadar layers)) (not (null? (cdr layers))))
	   (slib:warn 'removing 'layer (car layers))
	   (loop (cdr layers) layer stack))
	  (else (loop (cdr layers) (car layers) (cons layer stack)))))
  (if (null? stack)
      '((1 0) (1 0))
      (loop (if (null? (cdr stack)) '((1 0)) (cdr stack))
	    (car stack) '())))

(define (describe-stack layers)
  (define n1 (caar layers))
  (define (frac-wave frac)
    (define inv (/ 1 frac))
    (if (and (real? frac)
	     (< 1/50 frac 1)
	     (< (abs (- inv (inexact->exact (round inv)))) (* .01 inv)))
	(sprintf #f "1/%d" (inexact->exact inv))
	(sprintf #f "%.3f" frac)))
  (require 'printf)
  (cond ((number? n1) (printf "Top substrate IR = %g\\n" n1)))
  (for-each
   (lambda (layer)
     (if (number? (car layer))
	 (cond ((zero? (cadr layer))
		(printf "Bottom substrate IR = %g\\n" (car layer)))
	       ((null? (cddr layer))
		(printf "IR = %-11.3g%s%6.3Km thick (%6.3Km optical)\\n"
			(car layer)
			(if (negative? (cadr layer)) "~" " ")
			(abs (cadr layer))
			(* (abs (cadr layer)) (real-part (car layer)))))
	       (else
		(printf "IR = %-11.3g%s%6.3Km thick (%6.3Km optical) = %5s wave @ %6.3Km\\n"
			(car layer)
			(if (negative? (cadr layer)) "~" " ")
			(abs (cadr layer))
			(* (abs (cadr layer)) (real-part (car layer)))
			(frac-wave (/ (* (abs (cadr layer)) (real-part (car layer)))
				      (caddr layer)))
			(caddr layer))))
	 (if (null? (cddr layer))
	     (printf "IR ~ %-11s%s%6.3Km thick\\n" "?"
		     (if (negative? (cadr layer)) "~" " ")
		     (abs (cadr layer)))
	     (let ((rough ((car layer) (caddr layer))))
	       (printf "IR ~ %-11.3g%s%6.3Km thick (%6.3Km optical) ~ %5s wave @ %6.3Km\\n"
		       rough
		       (if (negative? (cadr layer)) "~" " ")
		       (abs (cadr layer))
		       (* (abs (cadr layer)) (real-part rough))
		       (frac-wave (/ (* (abs (cadr layer)) (real-part rough))
				     (caddr layer)))
		       (caddr layer))))))
   (cdr layers)))

(define (substrate index-of-refraction)
  (list 'substrate index-of-refraction))

(define (layer index-of-refraction thickness . nominal)
  (apply list 'layer index-of-refraction thickness nominal))

(define (layer* index-of-refraction thickness . nominal)
  (apply list 'layer index-of-refraction (- thickness) nominal))

;;(define (metallic index-of-refraction thickness)
;;  (list 'layer index-of-refraction thickness))

(define (nominal wl)
  (list 'nominal wl))

(define (repeat . args)
  (cons 'repeat args))
;;(trace-all "../FreeSnell/opticompute.scm") (set! *qp-width* 333)
