/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 16-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Set;
import java.io.IOException;


/**
 * <p>Classe principale di jrisk, un gioco di strategia ispirato
 * al famoso Risiko. Per far partire il gioco, crea un'istanza
 * di questa classe e chiama il metodo run().
 * Questa  la classe principale dell'applicazione, poiche
 * gestisce la creazione e la allocazione in memoria di tutte
 * le altre classi.</p>
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class Gioco {
    /** Numero minimo di partecipanti al gioco */
    protected final static int MIN_GIOCATORI = 2;
    /** Numero massimo di partecipanti al gioco */
    protected final static int MAX_GIOCATORI = 6;
    /** Numero di armate iniziali per ogni giocatore
     * in base al numero totale dei giocatori */
    protected final static int ARMATE_GIOCATORI[] = {
            40,
            35,
            30,
            25,
            20
    		};
    /** Colori disponibile per i giocatori */
    protected final static String COLORI[] = {
            "rosso",
            "nero",
            "verde",
            "blu",
            "viola",
            "giallo"
    };
    /** Elenco giocatori partecipanti*/
    protected List giocatori;
    /** Riferimento al planisfero del gioco */
    protected Tabellone planisfero;
    /** Riferimento ad un interpete di comandi */
    protected static Parser parser;
    /** Riferimento ad una periferica di visualizzazione */
    protected static Display schermo;
    
    /**
     * Inizializza la classe principale del gioco.
     */
    public Gioco() {
        schermo = new Display();
        parser = new Parser();
        giocatori = new LinkedList();
        planisfero = new Tabellone();
    }
    /**
     * Quando un giocatore non ha pi territori allora ha perso
     * e va eliminato dalla lista dei partecipanti al gioco.
     *
     */
    protected void eliminaGiocatoriSconfitti() {
        Iterator it = giocatori.iterator();
        Giocatore corrente;
        while(it.hasNext()) {
            corrente = (Giocatore) it.next();
            if(corrente.getTerritori().size()==0);
            	giocatori.remove(corrente);
        }
    }
    /**
     * Controlla se un giocatore ha vinto. Ci accade quando
     * rimane solo un giocatore nella lista dei partecipanti.
     *  il vincitore.
     * 
     * @return il riferimento al vincitore
     */
    protected Giocatore finePartita() {
        if(giocatori.size() == 1) {
            Iterator it = giocatori.iterator();
            return (Giocatore) it.next();
        }
        return null;
    }
    /**
     * Assegna i territori ai giocatori in maniera casuale.
     */
    protected void assegnaTerritori() {
        DistributoreTerritori distributoreTerritori;
        /* Distribuisce i territori a caso */
        distributoreTerritori = new DistributoreTerritori(planisfero);
        distributoreTerritori.assegnaCasualmente(giocatori);
        /* imposta i confini */
        distributoreTerritori.impostaConfini(giocatori);
    }
    /**
     * Dopo che sono stati distribuiti i territori, vengono distribuite
     * le armate in base al numero di partecipanti al gioco.
     * 
     * @throws <code>IOException</code>
     */
    protected void assegnaArmateIniziali() throws IOException {
        // pre: (giocatori != null) &&
        // (MIN_GIOCATORI <= giocatori.size() <= MAX_GIOCATORI)
        Giocatore g;								// giocatore corrente
        int n;										// numero partecipanti al gioco
        int armate;									// Numero armate da distribuire
        String territorio;							// Territorio dove posare l'armata
        Set territori;
        Territorio t;
        Iterator it,jt;
        int armateGiocatore;
        
        n = giocatori.size();						// Calcola il numero di giocatori
        /* Calcola quante sono le armate da distribuire */
        armate = ARMATE_GIOCATORI[n - MIN_GIOCATORI];
        
        /* Iteratore per i giocatori */
        it = giocatori.iterator();
        /* Ne mette un su ogni territorio (OBBLIGATORIO) */
        while(it.hasNext()) {
            g = (Giocatore) it.next();
            territori = g.getTerritori();
            jt = territori.iterator();
            while(jt.hasNext()) {
                t = (Territorio) jt.next();
                t.setArmate(Territorio.getMinimoArmate());
            }
        }
        /* Ad ogni giocatore deve far posizionare un numero
         * di armate pari alle armate iniziali meno il numero
         * di territori di suo possesso. La spiegazione di
         * quest'operazione  banale: se un giocatore
         * possiede 9 territori e ha a disposizione 25 armate
         * dovr posizionarne solo 16, dato che 9 sono
         * obbligatoriamente impostate sui territori occupati.
         */
        it = giocatori.iterator();
        while(it.hasNext()) {
            g = (Giocatore) it.next();
            armateGiocatore = armate - g.getTerritori().size();
            g.schermo.stampaln(g.toString());
            g.schermo.stampaln(g.getColore() + " ha " + armateGiocatore + " armate da posizionare. ");
            territorio = null;
            for(int i=0; i<armateGiocatore; i++) {
                if(territorio == null) {
                    g.schermo.stampa("Digita dove posizionare la " + (i+1) + " armata: ");
                    territorio = g.parser.getParola();
            	    // se il nome dello stato non  valido
            	    // ricomincia il while
                    if(ElencoTerritoriContinenti.territorioValido(territorio)) {
                	    // prendi il territorio richiesto dal giocatore
                        t = g.getTerritorio(territorio);
                        if(t != null) {
                            // sommagli un'armata
                            t.setArmate(t.getArmate()+1);
                            g.schermo.stampaln(t.getNome() + " ha " +t.getArmate() + " armate.");
                            territorio = null;
                        }
                        else {
                            i--;
                            territorio = null;
                        }
                    } else {									// se il nome non  valido
                        i--;
                        territorio = null;
                    }		
                }
            }
        }
        //System.out.println("---------------- ARMATE INIZIALI ASSEGNATE! DEBUG ----------------");
    }
    /**
     * TODO IN PARTE DA RISCRIVERE??
     * @throws <code>IOException</code>
     */
    protected void assegnaArmate(Giocatore g) throws IOException {
        // pre: (giocatori != null) &&
        // (MIN_GIOCATORI <= giocatori.size() <= MAX_GIOCATORI)
        int armate;									// Numero armate da distribuire
        String territorio;							// Territorio dove posare l'armata
        Territorio t;
        Iterator it;
        
        
        /* Calcola quante sono le armate da distribuire */
        armate = DistributoreArmate.daiArmate(g);
        
        g.schermo.stampaln(g.getColore() + " deve posizionare " + armate + " armate.");
        /* Distribuisce ogni armata */
        for(int i=0; i<armate; i++) {
            /* Prendi il giocatore con
             * pi alta precedenza
             */
            territorio = null;								// stringa vuota 
    	    // Chiede al giocatore dove vuole posizionare
        	// l'armata corrente
            if(territorio == null) {                    
                g.schermo.stampa("Digita dove posizionare la " + (i+1) +" armata: ");
                territorio = g.parser.getParola();
        	    // se il nome dello stato non  valido
        	    // ricomincia il while
                if(ElencoTerritoriContinenti.territorioValido(territorio)) {
            	    // prendi il territorio richiesto dal giocatore
                    t = g.getTerritorio(territorio);
                    if(t != null) {
                        // sommagli un'armata
                        t.setArmate(t.getArmate()+1);
                        g.schermo.stampaln(t.getNome() + " ha " +t.getArmate());
                        territorio = null;
                    }
                    else {
                        i--;
                        territorio = null;
                    }                    
                } else {									// se il nome non  valido
                    i--;
                    territorio = null;
                }											// poni di nuovo la domanda
            } else {
                i--;
                territorio = null;
            }
    	}
        //System.out.println("---------------- ARMATE ASSEGNATE! DEBUG ----------------");
    }
    
    /**
     * Crea e inizializza un numero desiderato di oggetti
     * Giocatore, uno per ogni partecipante al gioco.
     * 
     * @throws <code>IOException</code>
     */
    protected void impostaGiocatori() throws IOException {
        int n = 0;

        schermo.stampa("Quanti sono i giocatori (2 - 6): ");
        n = parser.getInt();
        while(n < MIN_GIOCATORI || n > MAX_GIOCATORI) {
            schermo.stampa("\nNumero errato. (2 - 6): ");
            n = parser.getInt();
        }
        
        for(int i=0; i<n; i++) {
            giocatori.add(new Giocatore(COLORI[i]));
        }
    }
    
    /**
     * Have fun!
     * 
     * @throws <code>IOException</code>
     */
    public void run() throws IOException {
        Giocatore vincitore = null;
        Iterator it;
        Giocatore toccaA;
        Comando comando;
        boolean passaTurno;
        //int turni = 0;
        
        /* Quanti giocatori partecipano? */
        impostaGiocatori();
        /* Assegna i territori a caso */
        assegnaTerritori();
        /* Stampa le informazioni sui giocatori */
        it = giocatori.iterator();
        while(it.hasNext()) {
            toccaA = (Giocatore) it.next();
            toccaA.schermo.stampaln(toccaA.toString());
        }
        /* Ogni territorio deve avere almeno un'armata */
        assegnaArmateIniziali();
        while(vincitore == null) {
            //turni++;			// Nuovo turno
            it = giocatori.iterator();
            Giocatore.aggiornaTurni();
            while(it.hasNext()) {
                /* Comincia il turno del giocatore */
                passaTurno = false;
                toccaA = (Giocatore) it.next();
                /* Assegna le armate */
                toccaA.schermo.stampaln(toccaA.toString());
                assegnaArmate(toccaA);
                while(!passaTurno) {
                /* prendi il comando */
                    comando = toccaA.parser.getComando();
                    if(comando.getNomeComando()=="esci") {
                        toccaA.schermo.stampaln("Sei sicuro di voler uscire? ");
                        if(toccaA.parser.getParola().equals("si"))
                            return;
                    }
                    passaTurno = comando.esegui(toccaA, planisfero);
                }
                if(toccaA.eliminato())
                    it.remove();
                else
                    if(toccaA.haVinto())
                        vincitore = toccaA;
                    else
                        if(DistributoreCarte.daiCarta(toccaA) == true)
                            schermo.stampaln(toccaA.getColore() + " pesca una carta...");
            }
        }
        schermo.stampa(vincitore.getColore() + " ha vinto in " +
                Giocatore.getTurnoCorrente() + " turni!");
    }
    
    /**
     * Restituisce l'elenco dei partecipanti.
     * 
     * @return l'elenco dei giocatori
     */
    public List getGiocatori() { return giocatori; }


    public static void main(String[] args) {
        Gioco jrisk = new Gioco();
        try {
            jrisk.run();
        } catch(IOException e) {
            schermo.stampaln("Errore fatale!");
        }
    }
}
