/*
 * @(#)MIBTreeModel.java       1.0           15 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import java.util.Enumeration;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import net.percederberg.mib.symbol.Symbol;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.type.SnmpTrapType;

/**
 * This class is a TreeModel interface to the MIB symbol tree.
 * It supports being inserted in TreeModel components, and makes
 * no changes to the symbols.
 *
 * @author   Per Cederberg, per@percederberg.net
 * @version  1.0
 */
public class MIBTreeModel extends Object implements TreeModel {

   /**
    * The root node.
    */
   private TreeSymbol   root;

   /**
    * The root node.
    */
   private TreeSymbol   traps;

   /**
    * The root node.
    */
   private TreeSymbol   types;

   /**
    * The enterprises node.
    */
   private Symbol       enterprises;

   /**
    * Creates a parse tree with the given parameters.
    *
    * @param mib      the MIB object
    */
   public MIBTreeModel(MIB mib) {
      Symbol   sym;

      // Sets the various root nodes
      this.root = new TreeSymbol(mib.toString());
      this.enterprises = mib.findSymbol("enterprises");
      if (this.enterprises != null) {
         this.root.addChild(this.enterprises);
      }
      this.traps = new TreeSymbol(this.root, "TRAPS");
      this.types = new TreeSymbol(this.root, "TEXTUAL CONVENTIONS");

      // Adds traps and types
      addTraps(mib);
      addTypes(mib);
   }

   /**
    * Adds all the trap symbols as children to the tree symbol for traps.
    *
    * @param   mib         the mib containing the symbols
    */
   private void addTraps(MIB mib) {
      Enumeration e = mib.allSymbols();
      Symbol      sym;

      while (e.hasMoreElements()) {
         sym = (Symbol)e.nextElement();
         if (sym.getType() instanceof SnmpTrapType) {
            this.traps.addChild(sym);
         }
      }
   }

   /**
    * Adds all the type symbols as children to the tree symbol for types.
    *
    * @param   mib         the mib containing the symbols
    */
   private void addTypes(MIB mib) {
      Enumeration e = mib.allSymbols();
      Symbol      sym;

      while (e.hasMoreElements()) {
         sym = (Symbol)e.nextElement();
         if (sym instanceof TypeSymbol) {
            this.types.addChild(sym);
         }
      }
   }

   /**
    * Returns the root of the tree.  Returns null only if the tree has
    * no nodes.
    *
    * @return  the root of the tree
    */
   public Object getRoot() {
      return this.root;
   }

   /**
    * Returns the child of <I>parent</I> at index <I>index</I> in the parent's
    * child array.  <I>parent</I> must be a node previously obtained from
    * this data source. This should not return null if <i>index</i>
    * is a valid index for <i>parent</i> (that is <i>index</i> >= 0 &&
    * <i>index</i> < getChildCount(<i>parent</i>)).
    *
    * @param parent  a node in the tree, obtained from this data source
    * @param index   the child index
    *
    * @return the child of <I>parent</I> at index <I>index</I>
    */
   public Object getChild(Object parent, int index) {
      return ((Symbol)parent).childAt(index);
   }

   /**
    * Returns the number of children of <I>parent</I>.  Returns 0 if the node
    * is a leaf or if it has no children.  <I>parent</I> must be a node
    * previously obtained from this data source.
    *
    * @param   parent  a node in the tree, obtained from this data source
    * @return  the number of children of the node <I>parent</I>
    */
   public int getChildCount(Object parent) {
      return ((Symbol)parent).children();
   }

   /**
    * Returns true if <I>node</I> is a leaf.  It is possible for this method
    * to return false even if <I>node</I> has no children.  A directory in a
    * filesystem, for example, may contain no files; the node representing
    * the directory is not a leaf, but it also has no children.
    *
    * @param   node    a node in the tree, obtained from this data source
    * @return  true if <I>node</I> is a leaf
    */
   public boolean isLeaf(Object node) {
      return getChildCount(node) == 0 && !(node instanceof TreeSymbol);
   }

   /**
    * Messaged when the user has altered the value for the item identified
    * by <I>path</I> to <I>newValue</I>.  If <I>newValue</I> signifies
    * a truly new value the model should post a treeNodesChanged
    * event.
    *
    * @param path path to the node that the user has altered.
    * @param newValue the new value from the TreeCellEditor.
    */
   public void valueForPathChanged(TreePath path, Object newValue) {
      // Ignored
   }

   /**
    * Returns the index of child in parent.
    *
    * @param parent  the parent node
    * @param child   the child node
    * @return  the child index, or -1 if not present
    */
   public int getIndexOfChild(Object parent, Object child) {
      int      i = 0;
      Symbol   symbol = (Symbol)parent;

      while (i < symbol.children()) {
         if (symbol.childAt(i) == child) {
            return i;
         }
         i++;
      }
      return -1;
   }

   /**
    * Adds a listener for the TreeModelEvent posted after the tree changes.
    *
    * @see     #removeTreeModelListener
    * @param   l       the listener to add
    */
   public void addTreeModelListener(TreeModelListener l) {
      // Ignored
   }

   /**
    * Removes a listener previously added with <B>addTreeModelListener()</B>.
    *
    * @see     #addTreeModelListener
    * @param   l       the listener to remove
    */
   public void removeTreeModelListener(TreeModelListener l) {
      // Ignored
   }
}


/**
 * A class for allowing symbol nodes that are not really symbols from
 * the MIB file. Typically three tree symbols will be created; the root,
 * the "TEXUAL CONVENTIONS" node and the "TRAPS" node.
 *
 * @author   Per Cederberg, per@percederberg.net
 */
class TreeSymbol extends Symbol {

   /**
    * Creates a new root tree symbol with the given name.
    *
    * @param   name     the root name (i.e. the file name)
    */
   public TreeSymbol(String name) {
      this.name = name;
   }

   /**
    * Creates a new tree symbol with the given parent and name.
    *
    * @param   parent     the parent symbol
    * @param   name       the symbol name
    */
   public TreeSymbol(Symbol parent, String name) {
      this.name = name;
      this.setParent(parent);
   }

   /**
    * Overrules the protected definition of addChild from symbol
    * in order to add children that have no registered parent.
    *
    * @param child    the child to add
    */
   public void addChild(Symbol child) {
      super.addChild(child);
   }

   /**
    * Returns an empty string.
    *
    * @return an empty string
    */
   public String getOID() {
      return "";
   }

   /**
    * Throws an exception, since this operation is not allowed for
    * tree symbols.
    *
    * @param id     the new OID
    *
    * @exception UnsupportedOperationException thrown as this
    *                operation is not supported
    */
   public void setOID(int id) throws UnsupportedOperationException {
      throw new UnsupportedOperationException("Cannot assign an id to " +
                                              this.toString());
   }
}

