/*
 * @(#)ValueRangeConstraint.java      1.0     6 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

/**
 * A class representing a value range constraint. Only the integer
 * values of the numbers contained will be used.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class ValueRangeConstraint extends Constraint {

    /**
     * The minimum unlimited value.
     */
    public static final Number  MIN = null;

    /**
     * The maximum unlimited value.
     */
    public static final Number  MAX = null;

    /**
     * The lower boundary value.
     */
    private Number lower;

    /**
     * The upper boundary value.
     */
    private Number upper;

    /**
     * Creates a new value range constraint from the given boundaries.
     * The value boundaries are considered inclusive.
     *
     * @param  lower    the lower boundary, or MIN for unlimited
     * @param  upper    the upper boundary, or MAX for unlimited
     */
    public ValueRangeConstraint(Number lower, Number upper) {
        this(lower, true, upper, true);
    }

    /**
     * Creates a new value range constraint from the given boundaries.
     * The value boundaries can be made either inclusive or exclusive.
     * Note that in the latter case a value of one is just added or
     * subtracted to the value.
     *
     * @param  lower    the lower boundary, or MIN for unlimited
     * @param  lowinc   true if lower boundary inclusive,
     *                  false otherwise
     * @param  upper    the upper boundary, or MAX for unlimited
     * @param  upinc    true if upper boundary inclusive,
     *                  false otherwise
     */
    public ValueRangeConstraint(Number lower, boolean lowinc,
                                Number upper, boolean upinc) {

        if (lowinc) {
            this.lower = lower;
        } else if (lower == null) {
            this.lower = MIN;
        } else {
            this.lower = new Long(lower.longValue() + 1);
        }
        if (upinc) {
            this.upper = upper;
        } else if (upper == null) {
            this.upper = MIN;
        } else {
            this.upper = new Long(upper.longValue() + 1);
        }
    }

    /**
     * Checks if this constraint equals another constraint.
     *
     * @param obj    the object to compare with
     *
     * @return true if the objects are equal, or
     *         false otherwise
     */
    public boolean equals(Object obj) {
        if (obj instanceof ValueRangeConstraint) {
            ValueRangeConstraint c = (ValueRangeConstraint) obj;
            return equalsValue(this.lower, c.lower) &&
                equalsValue(this.upper, c.upper);
        } else {
            return false;
        }
    }

    /**
     * Returns a string description of this constraint.
     *
     * @return a string description
     */
    public String toString() {
        String res = "";
        if (lower == MIN) {
            res = res + "MIN";
        } else {
            res = res + lower.longValue();
        }
        res = res + "..";
        if (upper == MAX) {
            res = res + "MAX";
        } else {
            res = res + upper.longValue();
        }
        return res;
    }

    /**
     * Transfers the constraint information from this object to a type
     * converter object. The calling conventions declared in the
     * TypeConverter class are followed.
     *
     * @param   converter     a type converter
     */
    public void transferConstraint(TypeConverter converter) {
        converter.transferValueLimits(minValue(), maxValue());
    }

    /**
     * Returns the minimum integer value for this constraint.
     *
     * @return   the minium integer value
     */
    protected int minValue() {
        if (lower == MIN || lower.longValue() < Integer.MIN_VALUE) {
            return Integer.MIN_VALUE;
        } else if (lower.longValue() > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        } else {
            return lower.intValue();
        }
    }

    /**
     * Returns the maximum integer value for this constraint.
     *
     * @return   the maximum integer value
     */
    protected int maxValue() {
        if (upper == MAX || upper.longValue() > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        } else if (upper.longValue() < Integer.MIN_VALUE) {
            return Integer.MIN_VALUE;
        } else {
            return upper.intValue();
        }
    }

}
