/*
 * @(#)ValueSymbol.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.symbol;

import net.percederberg.mib.type.Type;
import net.percederberg.mib.type.SnmpObjectType;

/**
 * A class containing a value symbol from the MIB file. Each value
 * symbol should be given a type and optionally a parent and an
 * object identifier.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class ValueSymbol extends Symbol {

    /**
     * Creates a new value symbol.
     *
     * @param   name     the symbol name
     */
    public ValueSymbol(String name) {
        this.name = name;
    }

    /**
     * Creates a new value symbol with the given parameters.
     *
     * @param   name      the symbol name
     * @param   type      the symbol type
     * @param   id        the object identifier number
     */
    public ValueSymbol(String name, Type type, int id) {
        this.name = name;
        this.type = type;
        this.id = id;
    }

    /**
     * Creates a new value symbol with the given parameters.
     *
     * @param   name      the symbol name
     * @param   type      the symbol type
     * @param   parent    the parent symbol
     * @param   id        the object identifier number
     */
    public ValueSymbol(String name, Type type, Symbol parent, int id) {
        this.name = name;
        this.type = type;
        this.id = id;
        this.setParent(parent);
    }

    /**
     * Checks if this symbol represents a data field in the model.
     *
     * @return true if this symbol represents a data field, or
     *         false otherwise
     */
    public boolean isField() {
        return this.type instanceof SnmpObjectType;
    }

    /**
     * Checks if this symbol represents a top data field in the
     * hierarchy. Typically data fields contained within arrays and
     * composed types will return false for this method.
     *
     * @return true if this symbol represents a top data field, or
     *         false otherwise
     */
    public boolean isTopField() {
        return isField() && !this.parent.isField();
    }
}
