/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \file MLV_prompt.h
 *
 * \author Adrien Boussicault
 * \author Marc Zipstein
 *
 * Ce fichier définit tous les prototypes des fonctions permettant de demander à
 * l'utilisateur du texte par l'intermédiaire de fenêtres.
 *
 */

#ifndef __MLV_PROMPT_H__
#define __MLV_PROMPT_H__

#include "MLV_color.h"

/**
 * \brief Définition du type des prompts
 *
 */
typedef struct _MLV_Prompt MLV_Prompt;

/**
 *
 * \brief Suspend l'exécution du programme et demande du texte à l'utilisateur.
 *
 * Suspend l'exécution du programme et affiche deux zones de textes. 
 * La première zone n'est pas modifiable et sert à afficher un
 * message d'information. La deuxième est vide et peut être modifiée
 * par l'utilisateur.
 * Lorsque l'utilisateur appuie sur la touche entrée, le texte qui 
 * se trouve dans la zone de texte est renvoyé par la fonction
 * grâce au paramètre text.
 * Le texte, la taille et les couleurs des différents éléments sont
 * paramétrables.
 *
 * \param sommetHautGaucheX     Coordonnée en X du coin Nord-Ouest du prompt
 * \param sommetHautGaucheY     Coordonnée en Y du coin Nord-Ouest du prompt
 * \param sommetBasDroitX       Coordonnée en X du coin Sud-Est du prompt
 * \param sommetBasDroitY       Coordonnée en Y du coin Sud-Est du prompt
 * \param borderColor           Couleur de la bordure du prompt
 * \param textColor             Couleur du texte du prompt
 * \param backgroundColor       Couleur de fond du prompt
 * \param informativeMessage    Message à afficher devant le prompt
 * \param text                  Addresse du texte où se trouvera la réponse donnée par l'utilisateur.
 */
void MLV_prompt_wait(
	int sommetHautGaucheX, int sommetHautGaucheY,
	int sommetBasDroitX, int sommetBasDroitY,
	MLV_Color borderColor, MLV_Color textColor,
	MLV_Color backgroundColor,
	const char* informativeMessage,
	char** text
);

/**
 * \brief Suspend l'exécution du programme jusuq'à ce que l'utilisateur valide un texte dans le prompt passé en paramètre de la fonction.
 *
 * \param prompt prompt qui doit être observé.
 * \param text text récupéré par le prompt.
 */
void MLV_wait_particular_prompt( MLV_Prompt* prompt, char** text);


/**
 * \brief Créé un prompt
 *
 * Une fois le prompt créé, si l'utilisateur clique dans la zone du prompt, le
 * prompt devient actif.
 * Cela veux dire que tout les évènements clavier sont récupérés par le prompt
 * qui s'en sert pour enregistrer le texte tapé par l'utilisateur.
 * De même, tout clique de souris réalisé dans la zonne du prompt est acaparé par
 * le prompt.
 *
 * Pour dessiner un prompt ou tous les prompts, dans la zone où vous l'avez 
 * créé, vous devez utiliser les fonctions : 
 *     MLV_draw_prompt
 *     MLV_draw_all_prompt
 *
 * \param sommetHautGaucheX Coordonnée en X du sommet Nord-Ouest du prompt.
 * \param sommetHautGaucheY Coordonnée en Y du sommet Nord-Ouest du prompt.
 * \param width Largeur du prompt
 * \param height Hauteur du prompt
 * \param borderColor Couleur de la bordure du prompt.
 * \param textColor Couleur du texte.
 * \param backgroundColor Couleur du fond.
 * \param informativeMessage Message d'entête du prompt.
 */
MLV_Prompt* MLV_create_prompt(
	int sommetHautGaucheX, int sommetHautGaucheY,
	int width, int height,
	MLV_Color borderColor, MLV_Color textColor,
	MLV_Color backgroundColor,
	const char* informativeMessage
);
/**
 * \brief Ferme un prompt précédement créé.
 *
 * \param prompt Le prompt à fermer.
 */
void MLV_close_prompt( MLV_Prompt* prompt );
/**
 * \brief Change le message d'entête d'un prompt.
 *
 * \param prompt Prompt à modifier.
 * \param message Message d'entête.
 */
void MLV_change_informative_message_prompt(
	MLV_Prompt* prompt, const char* message
);
/**
 * \brief Dessine un prompt particulier.
 *
 * \param prompt Le prompt à dessiner.
 */
void MLV_draw_prompt(MLV_Prompt* prompt);
/**
 * \brief Dessine tous les prompts.
 *

 */
void MLV_draw_all_prompt();
/**
 * \brief Supprime l'historique d'un prompt.
 *
 * \param prompt Le prompt dont l'histoirque doit être supprimé
 */
void MLV_suppress_history( MLV_Prompt* prompt );
/**
 * \brief Change la taille et la position d'un prompt donné en paramètre.
 *
 * \param prompt Le prompt à modifier
 * \param sommetHautGaucheX La nouvelle coordonnée en X de la position du sommet Nord-Ouest du prompt.
 * \param sommetHautGaucheY La nouvelle coordonnée en Y de la position du sommet Nord-Ouest du prompt.
 * \param width La nouvelle largeur du prompt
 * \param height La nouvelle hauteur du prompt
 */
void MLV_change_geometry_prompt(
	MLV_Prompt* prompt,  int sommetHautGaucheX, int sommetHautGaucheY,
	int width, int height
);
/**
 * \brief change la taille d'un prompt
 *
 * \param prompt Le prompt à modifier.
 * \param width La nouvelle largeur du prompt
 * \param height La nouvelle hauteur du prompt
 */
void MLV_change_size_prompt(MLV_Prompt* prompt, int width, int height);
/**
 * \brief Change la position d'un prompt donné en paramètre.
 *
 * \param prompt L prompt à modifier
 * \param sommetHautGaucheX La nouvelle coordonnée en X de la position du sommet Nord-Ouest du prompt.
 * \param sommetHautGaucheY La nouvelle coordonnée en Y de la position du sommet Nord-Ouest du prompt.
 */
void MLV_change_position_prompt(
	MLV_Prompt* prompt, int sommetHautGaucheX, int sommetHautGaucheY
);
/**
 * \brief Change les différentes couleurs d'un prompt donné
 *
 * \param prompt Le prompt à modifier.
 * \param borderColor La nouvelle couleur du bord du prompt.
 * \param textColor La nouvelle couleur du texte du prompt.
 * \param backgroundColor La nouvelle couleur de fond du prompt.
 */
void MLV_change_colors_prompt( 
	MLV_Prompt* prompt,  MLV_Color borderColor, MLV_Color textColor,
	MLV_Color backgroundColor
);
#endif
