// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief  C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "imagefile.h"
#include "imagefile_dossier.h"
#include "imagefile_ewf.h"
#include "imagefile_msr.h"
#include "imagefile_raw.h"
#include "imagefile_solo.h"
#include "imagefile_split.h"
#include "imagefile_talon.h"
#include "imagefile_vhd.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Function new_imagefile_from_url
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject*
func_new_imagefile_from_url (PyObject *, PyObject *args)
{
  // parse arguments
  const char *arg_url = nullptr;
  const char *arg_type = "autodetect";

  if (!PyArg_ParseTuple (args, "s|s", &arg_url, &arg_type))
    return nullptr;

  // execute C++ code
  mobius::imagefile::imagefile imagefile;
  PyThreadState *_save = PyEval_SaveThread ();

  try
    {
      imagefile = mobius::imagefile::new_imagefile_from_url (arg_url, arg_type);
      PyEval_RestoreThread(_save);
    }
  catch (const std::runtime_error& e)
    {
      PyEval_RestoreThread(_save);
      PyErr_SetString (PyExc_IOError, e.what ());
      return nullptr;
    }

  // create Python imagefile according to its type
  PyObject *ret = nullptr;

  if (imagefile.get_type () == "dossier")
    ret = PyMobius_Imagefile_Imagefile_Dossier_from_cpp (mobius::imagefile::imagefile_dossier (imagefile));

  else if (imagefile.get_type () == "ewf")
    ret = PyMobius_Imagefile_Imagefile_Ewf_from_cpp (mobius::imagefile::imagefile_ewf (imagefile));

  else if (imagefile.get_type () == "msr")
    ret = PyMobius_Imagefile_Imagefile_Msr_from_cpp (mobius::imagefile::imagefile_msr (imagefile));

  else if (imagefile.get_type () == "raw")
    ret = PyMobius_Imagefile_Imagefile_Raw_from_cpp (mobius::imagefile::imagefile_raw (imagefile));

  else if (imagefile.get_type () == "solo")
    ret = PyMobius_Imagefile_Imagefile_Solo_from_cpp (mobius::imagefile::imagefile_solo (imagefile));

  else if (imagefile.get_type () == "split")
    ret = PyMobius_Imagefile_Imagefile_Split_from_cpp (mobius::imagefile::imagefile_split (imagefile));

  else if (imagefile.get_type () == "talon")
    ret = PyMobius_Imagefile_Imagefile_Talon_from_cpp (mobius::imagefile::imagefile_talon (imagefile));

  else if (imagefile.get_type () == "vhd")
    ret = PyMobius_Imagefile_Imagefile_Vhd_from_cpp (mobius::imagefile::imagefile_vhd (imagefile));

  else
    ret = PyMobius_Imagefile_Imagefile_from_cpp (imagefile);

  // return data
  return ret;
}
