#   Copyright (C) 2002 Yannick Gingras <ygingras@ygingras.net>
#   Copyright (C) 2002 Vincent Barbin <vbarbin@openbeatbox.org>

#   This file is part of Open Beat Box.

#   Open Beat Box is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.

#   Open Beat Box is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.

#   You should have received a copy of the GNU General Public License
#   along with Open Beat Box; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from qt import *
from SimpleApp import SimpleApp
import time
from threading import Thread
from OBBGui.Floater import Floater
from OBBGui.OBBWidget import OBBWidget
from OBBGui.PushButton import PushButton
from OBBGui.ToggleButton import ToggleButton
from OBBGui.OBBLight import OBBLight
from OBBGui.OBBFrame import OBBFrame
from OBBGui.OBBText import OBBText
from OBBGui.PixmapSet import *
from OBBFuncts import *
import os
import sys

# sound stuff
from OBBSound.OBBSongCreator import OBBSongCreator
from OBBSound.OBBSongDocument import OBBSongDocument
from OBBSound.SoundDevice import SoundDevice
from OBBSound.PosixSoundDevice import PosixSoundDevice

MAX_LED = 8
SPLASH_DELAY = 5

class PatEditorDemo(SimpleApp):
    def __init__( self, demoDelay ):
        SimpleApp.__init__(self)
        self.demoDelay = demoDelay
        self.hitTimer = QTimer(self)
        self.curHitId = 0
        self.lambdaFuncts = [] # to keep a ref on anonymous functs
        self.connect(self.hitTimer, SIGNAL("timeout()"), self.cycleHits)
        self.scrollTimer = QTimer(self)
        self.connect(self.scrollTimer, SIGNAL("timeout()"), self.scroll)

    def exec_loop(self):
        self.app = QApplication([])
        #self.connect(self.app, SIGNAL("guiThreadAwake()"), self.startDelay)

        print "delay : %d" % self.demoDelay
        self.quitTimer = QTimer()

        self.initSound()
        self.initWidgets()

        if self.demoDelay: # auto-kick gui (probably in a unittest)
            self.quitTimer.singleShot( self.demoDelay * 1000,
                                       self.quit )
        else: # show the ugly splash ! : )
            self.initSplash()
            self.quitTimer.singleShot( SPLASH_DELAY * 1000,
                                       self.kickSplash )

        self.scrollTimer.start(50)
        self.app.exec_loop()

    def initSound(self):
        self.doc = OBBSongDocument()
        self.doc.SetTempo(428)
        self.creator = OBBSongCreator()
        self.sndDev = SoundDevice()

        self.hitDelay = self.doc.GetTempo()

        self.creator.AddInstrument( self.doc,
                                    0,
                                    os.path.join( getSndDir(),
                                                  "hat03.wav") )

        self.creator.AddInstrument( self.doc,
                                    1,
                                    os.path.join( getSndDir(),
                                                  "hat07.wav") )

        self.creator.AddInstrument( self.doc,
                                    2,
                                    os.path.join( getSndDir(),
                                                  "kick01.wav") )

        self.creator.AddInstrument( self.doc,
                                    3,
                                    os.path.join( getSndDir(),
                                                  "snr02.wav") )
        
        self.connect( self, PYSIGNAL("hit(int)"), self.sndDev.HandleHit )
        self.connect( self, PYSIGNAL("hit(int)"), self.forwardHit )
        self.connect( self,
                      PYSIGNAL("hitStateChanged(int, int)"),
                      self.forwardHitState )
        self.connect( self, PYSIGNAL("play()"), self.playLoop )

        # HandleLoadSample(samplePath):
        self.connect( self.creator,
                      PYSIGNAL("createSample()"),
                      self.sndDev.HandleLoadSample )

    def forwardHit(self, hitId):
        self.creator.HandleHit(self.doc, hitId)
    
    def forwardHitState(self, holderId, hitId):
        self.creator.HandleSongChange(self.doc, holderId, hitId)

    def playLoop(self):
        self.sndDev.HandleHit(7)
        self.forwardHit(7)

    def kickSplash(self):
        del(self.splash)

    def initSplash(self):
        self.splash = Floater()

        splashFrame = OBBFrame(self.splashPix, self.splash)
        
        self.splash.addSubWidget(splashFrame)
        self.splash.reCenter()
        self.splash.show()

    def changeHitState(self, holderId, buttonId):
        print "stateChanged : (%d, %d)" % ( holderId, buttonId )
        self.emit(PYSIGNAL("hitStateChanged(int, int)"), ( holderId,
                                                           buttonId ))    

    def play(self):
        print "play"
        self.curHitId = 7
        self.emit(PYSIGNAL("stop()"), ())
        self.hitTimer.start(self.hitDelay)
        self.emit(PYSIGNAL("play()"), ())

    def rec(self):
        print "rec"
        self.emit(PYSIGNAL("rec()"), ())

    def stop(self):
        print "stop"
        self.hitTimer.stop()
        self.emit(PYSIGNAL("stop()"), ())
        
    def quit(self):
        print "quit"
        self.stop()
        self.app.closeAllWindows()

    def cycleHits(self):
        self.curHitId = (self.curHitId + 1) % MAX_LED
        print "hit on %d" % self.curHitId
        self.emit(PYSIGNAL("hit(int)"), (self.curHitId,))

    def updateLedState(self, led, hitId, curHit):
        if hitId == curHit:
            led.turnOn()
        elif (hitId+1) % MAX_LED == curHit: # previous led
            led.turnOff()
        else:
            pass # nothing to do

    def scroll(self):
        self.emit(PYSIGNAL("scroll()"), ())

    def loadButtonPixmaps(self, type):
        pixmaps = PixmapSet()
        for state in (DISABLED, ACTIVATED, DESACTIVATED):
            pixmap = QPixmap( os.path.join( getImgDir(),
                                            "%s_%s.png" % (type, state)))
            pixmaps.addState(pixmap, state)
        return pixmaps

    def loadSimplePixmaps(self, type):
        pixmaps = PixmapSet()
        pixmap = QPixmap( os.path.join( getImgDir(),
                                        ("%s.png" % type)))
        pixmaps.addState(pixmap, DISABLED)
        
        return pixmaps

        
    def addPushButton(self, pixmapSet, parent, x, y, command):
        button = PushButton(pixmapSet, parent, x, y)
        self.connect(button, PYSIGNAL("clicked()"), command)
        parent.addSubWidget(button)


    def addToggleButton(self, pixmapSet, parent, x, y, holderId, butId):
        button = ToggleButton(pixmapSet, parent, x, y)
        func = lambda hId=holderId, bId=butId : self.changeHitState(hId, bId)
        self.lambdaFuncts.append(func)
        self.connect( button,
                      PYSIGNAL("stateChanged()"),
                      func )
        parent.addSubWidget(button)

    def addLed(self, pixmapSet, parent, x, y, holderId):
        led = OBBLight(pixmapSet, parent, x, y)
        
        func = lambda curHit, wid=led, hId=holderId: self.updateLedState(
            wid,
            hId,
            curHit )
        self.lambdaFuncts.append(func)
        self.connect(self, PYSIGNAL("hit(int)"), func)
        self.connect(self, PYSIGNAL("stop()"), led.turnOff)
        
        parent.addSubWidget(led)


    def loadImages(self):
        self.splashPix = self.loadSimplePixmaps("splash")
        
        self.buttonPixmaps = self.loadButtonPixmaps("hit")
        self.ledPixmaps = self.loadButtonPixmaps("led")
        self.recPixmaps = self.loadButtonPixmaps("rec")
        self.playPixmaps = self.loadButtonPixmaps("play")
        self.stopPixmaps = self.loadButtonPixmaps("stop")
        self.powPixmaps = self.loadButtonPixmaps("pow")

        self.vholderPixmaps = self.loadSimplePixmaps("vholder")
        self.fontPixmaps = self.loadSimplePixmaps("font")
        self.holderPixmaps = self.loadSimplePixmaps("holder")

    def initWidgets(self):

        self.prtFuncts = []
        self.loadImages()
        self.floater = Floater()
        vholder = OBBFrame(self.vholderPixmaps, self.floater)
        self.floater.addSubWidget(vholder)

        # holders
        for i in range(4):
            holder = OBBFrame(self.holderPixmaps, self.floater, 25, i*130+15)

            # only one label (they are too slow)
            if i == 0:
                label = OBBText(self.fontPixmaps, holder, 155, 22)
                self.connect(self, PYSIGNAL("scroll()"), label.scroll)
                holder.addSubWidget(label)

            self.addPushButton(self.powPixmaps, holder,  27, 51, self.quit)
            self.addPushButton(self.playPixmaps, holder,  60, 17, self.play)
            self.addPushButton(self.recPixmaps,  holder,  92, 51, self.rec)
            self.addPushButton(self.stopPixmaps,  holder, 125, 17, self.stop)

            # buttons
            for j in range(MAX_LED/2,MAX_LED):
                self.addToggleButton( self.buttonPixmaps,
                                      holder,
                                      j*100+110,
                                      15,
                                      i,
                                      j-4)

                self.addLed(self.ledPixmaps, holder, j*100+185, 37, j-4)

            for j in range(MAX_LED/2,MAX_LED):
                self.addToggleButton( self.buttonPixmaps,
                                      holder,
                                      j*100+70,
                                      63,
                                      i,
                                      j )

                self.addLed(self.ledPixmaps, holder, j*100+145, 85, j)

            self.floater.addSubWidget(holder)

        #self.floater.addSubWidget(vholder)
        self.floater.reCenter()
        self.floater.show()

        self.app.setMainWidget(self.floater)
        
