/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */

package opale.m2d;
import java.awt.geom.AffineTransform;
import opale.tools.*;
import java.io.*;


/**
* This class extends <code>java.awt.geom.AffineTransform</code> and represents a 2D affine transform. 
* @see <code>java.awt.geom.AffineTransform</code>
* @author O.C.
* @since Opale-2d 0.12
* @version 0.1
*/

public final class AffineTransform2D  extends Object2D 
{
private AffineTransform afft;

/**
 * Constructs a new <code>AffineTransform2D</code> representing the
 * Identity transformation.
 */
public AffineTransform2D() {
   super();
   afft = new AffineTransform();
}

/**
* Constructs a new <code>AffineTransform2D</code> representing the
* Identity transformation in a specified basis.
* @param OIJ rep
*/
public AffineTransform2D(OIJ rep) {
   super(rep);
   afft = new AffineTransform();
}

/**
 * Constructs a new <code>AffineTransform2D</code> that is a copy of
 * the specified <code>AffineTransform2D</code> object.
 * @param Tx the <code>AffineTransform2D</code> object to copy 
 */
public AffineTransform2D(AffineTransform2D Tx) {
   super(Tx.getOIJ());
    afft = new AffineTransform(Tx.afft);
}

/**
 * Constructs a new <code>AffineTransform2D</code> from 6 double
 * precision values representing the 6 specifiable entries of the 3x3
 * transformation matrix.
 * @param m00,&nbsp;m01,&nbsp;m02,&nbsp;m10,&nbsp;m11,&nbsp;m12 the 
 * 6 floating point values that compose the 3x3 transformation matrix
 */
public AffineTransform2D(double m00, double m10,
    		       double m01, double m11,
    		       double m02, double m12) {
    super();
    afft = new AffineTransform(m00,m10,m01,m11,m02,m12);
}


/**
 * Constructs a new <code>AffineTransform2D</code> from an array of
 * double precision values representing either the 4 non-translation
 * entries or the 6 specifiable entries of the 3x3 transformation
 * matrix. The values are retrieved from the array as 
 * {&nbsp;m00&nbsp;m10&nbsp;m01&nbsp;m11&nbsp;[m02&nbsp;m12]}.     
 * @param flatmatrix the double array containing the values to be set
 * in the new <code>AffineTransform</code> object. The length of the
 * array is assumed to be at least 4. If the length of the array is 
 * less than 6, only the first 4 values are taken. If the length of
 * the array is greater than 6, the first 6 values are taken.
 */
public AffineTransform2D(double[] tab) {
    super();
    afft = new AffineTransform(tab);
}


/**
 * Retrieves the 6 specifiable values in the 3x3 affine transformation
 * matrix and places them into an array of double precisions values.
 * @param flatmatrix the double array used to store the returned
 * values.
 * @see #getScaleX
 * @see #getScaleY
 * @see #getShearX
 * @see #getShearY
 * @see #getTranslateX
 * @see #getTranslateY
 */
public void getMatrix(double[] flatmatrix) {
 afft.getMatrix(flatmatrix);
 }

/**
 * Returns the X coordinate scaling element (m00) of the 3x3
 * affine transformation matrix.
 * @return a double value that is the X coordinate of the scaling
 *  element of the affine transformation matrix.
 * @see #getMatrix
 */
public double getScaleX() {
    return afft.getScaleX();
}

/**
 * Returns the Y coordinate scaling element (m11) of the 3x3
 * affine transformation matrix.
 * @return a double value that is the Y coordinate of the scaling
 *  element of the affine transformation matrix.
 * @see #getMatrix
 */
public double getScaleY() {
    return afft.getScaleY();
}

/**
 * Returns the X coordinate shearing element (m01) of the 3x3
 * affine transformation matrix.
 * @return a double value that is the X coordinate of the shearing
 *  element of the affine transformation matrix.
 * @see #getMatrix
 */
public double getShearX() {
    return afft.getShearX();
}

/**
 * Returns the Y coordinate shearing element (m10) of the 3x3
 * affine transformation matrix.
 * @return a double value that is the Y coordinate of the shearing
 *  element of the affine transformation matrix.
 * @see #getMatrix
 */
public double getShearY() {
    return afft.getShearY();
}

/**
 * Returns the X coordinate of the translation element (m02) of the
 * 3x3 affine transformation matrix.
 * @return a double value that is the X coordinate of the translation
 *  element of the affine transformation matrix.
 * @see #getMatrix
 */
public double getTranslateX() {
    return afft.getTranslateX();
}

/**
 * Returns the Y coordinate of the translation element (m12) of the
 * 3x3 affine transformation matrix.
 * @return a double value that is the Y coordinate of the translation
 *  element of the affine transformation matrix. 
 * @see #getMatrix
 */
public double getTranslateY() {
    return afft.getTranslateY();
}



/**
 * Concatenates this transform with a translation transformation defined in the specified basis of this transformation.
 * @param tx the distance by which coordinates are translated in the
 * X axis direction
 * @param ty the distance by which coordinates are translated in the
 * Y axis direction
 */
public void translate(double tx, double ty) {
 afft.translate(tx,ty);

}

/**
 * Concatenates this transform with a scaling transformation defined in the specified basis of this transformation.
 * @param sx the factor by which coordinates are scaled along the   
 * X axis direction
 * @param sy the factor by which coordinates are scaled along the
 * Y axis direction 
*/
public void scale(double sx, double sy) {
 afft.scale(sx,sy);
}


/**
 * Concatenates this transform with a rotation transformation.
 * This is equivalent to calling concatenate(R), where R is an
 * <code>AffineTransform</code> represented by the following matrix:
 * <pre>
 *	    [	cos(alpha)-cos(theta)*sin(alpha)/sin(theta)    -sin(alpha)/normI/sin(theta)    0   ]
 *	    [	sin(alpha)/normJ/sin(theta)     cos(alpha)+cos(theta)*sin(alpha)/sin(theta)    0   ]
 *	    [	    0		   0	     1   ]
 * </pre>
 * where theta is the angle between the vector I and J of the definition basis.
 * Rotating with a positive angle theta rotates points on the positive
 * x axis toward the positive y axis.
 * @param theta the angle of rotation in radians
 */
public void rotate(double alpha) {
    final double costh=Math.cos(alpha);
    final double sinth=Math.sin(alpha);
    final OIJ rep = getOIJ();
    final double invsin = 1./Math.sin(rep.getTheta());
    final double tmp = Math.cos(rep.getTheta())*invsin*sinth;
    final double nI = Math.sqrt(rep.normSqI());
    final double nJ = Math.sqrt(rep.normSqJ());
    final double a00 =  costh - tmp;
    final double a01 =  -sinth*invsin/nI;
    final double a10 =   sinth*invsin/nJ;
    final double a11 =  costh + tmp;

    double[] mat = new double[6];
    afft.getMatrix(mat);
    final double m00 = a00*mat[0]+a10*mat[2];
    final double m01 = a01*mat[0]+a11*mat[2];
    final double m10 = a00*mat[1]+a10*mat[3];
    final double m11 = a01*mat[1]+a11*mat[3];

    afft.setTransform(m00,m10,m01,m11,mat[4],mat[5]);

}

/**
 * Concatenates this transform with a transform that rotates
 * coordinates around an anchor point.
 * @param theta the angle of rotation in radians
 * @param x,&nbsp;y the coordinates of the anchor point of the
 * rotation
 */
public void rotate(double theta, double x, double y) {
    translate(x, y);
    rotate(theta);
    translate(-x, -y);
}

/**
 * Concatenates this transform with a shearing transformation.
 * @param shx the multiplier by which coordinates are shifted in the
 * direction of the positive X axis as a factor of their Y coordinate
 * @param shy the multiplier by which coordinates are shifted in the
 * direction of the positive Y axis as a factor of their X coordinate
 */
public void shear(double shx, double shy) {
System.err.println("shear transformation not available for the moment !!!");
System.exit(-1);

}


/**
* Resets this transform to the Identity transform.
*/
public void setToIdentity() {
	afft.setToIdentity();
}

/**
 * Sets this transform to a translation transformation in its definition basis.
 * @param tx the distance by which coordinates are translated in the
 * X axis direction
 * @param ty the distance by which coordinates are translated in the
 * Y axis direction
 */
public void setToTranslation(double tx, double ty) {
afft.setToTranslation(tx, ty);
}

/**
 * Sets this transform to a scaling transformation.
 * @param sx the factor by which coordinates are scaled along the
 * X axis direction
 * @param sy the factor by which coordinates are scaled along the
 * Y axis direction
 */
public void setToScale(double sx, double sy) {
afft.setToScale(sx,sy);
}


/**
 * Sets this transform to a rotation transformation.
 * The matrix representing this transform becomes:
 * <pre>
 *	    [	cos(alpha)-cos(theta)*sin(alpha)/sin(theta)    -sin(alpha)/normI/sin(theta)    0   ]
 *	    [	sin(alpha)/normJ/sin(theta)     cos(alpha)+cos(theta)*sin(alpha)/sin(theta)    0   ]
 *	    [	    0		   0	     1   ]
 * </pre>
 * where theta is the angle between the vector I and J of the definition basis.
 * Rotating with a positive angle theta rotates points on the positive
 * x axis toward the positive y axis.
 * @param theta the angle of rotation in radians
 */
public void setToRotation(double alpha) {
    final double costh=Math.cos(alpha);
    final double sinth=Math.sin(alpha);
    final OIJ rep = getOIJ();
    final double invsin = 1./Math.sin(rep.getTheta());
    final double tmp = Math.cos(rep.getTheta())*invsin*sinth;
    final double nI = Math.sqrt(rep.normSqI());
    final double nJ = Math.sqrt(rep.normSqJ());
    final double a00 =  costh - tmp;
    final double a01 =  -sinth*invsin/nI;
    final double a10 =   sinth*invsin/nJ;
    final double a11 =  costh + tmp;


    afft.setTransform(a00,a10,a01,a11,0.,0.);

}

/**
 * Sets this transform to a translated rotation transformation.
 * @param theta the angle of rotation in radians
 * @param x,&nbsp;y the coordinates of the anchor point of the
 * rotation
 */
public void setToRotation(double theta, double x, double y) {
// No optimizations for the moment
if (Debug.On) Debug.print("No optimizations for the moment in AffineTransform2D::setToRotation");
setToTranslation(x, y);
rotate(theta);      
translate(-x, -y);  
}


/**
 * Sets this transform to a copy of the transform in the specified
 * <code>AffineTransform2D</code> object.
 * @param Tx the <code>AffineTransform2D</code> object from which to
 * copy the transform
 */
public void setTransform(AffineTransform2D Tx) {
afft.setTransform(Tx.afft);
}

/**
 * Sets this transform to the matrix specified by the 6
 * double precision values.
 * @param m00,&nbsp;m01,&nbsp;m02,&nbsp;m10,&nbsp;m11,&nbsp;m12 the
 * 6 floating point values that compose the 3x3 transformation matrix
 */
public void setTransform(double m00, double m10,
    			 double m01, double m11,
    			 double m02, double m12) {
afft.setTransform(m00,m10,m01,m11,m02,m12);
}

/**
 * Transforms  a point represented by two coordinates .
 * @param src the array containing the two source point coordinates .
 * @param dst the array into which are returned the two  transformed point coordinates.
 */
public void transform(double[] src, double[]  dst) {
    double[] mat = new double[6];
    afft.getMatrix(mat);
    	    dst[0] = mat[0] * src[0] + mat[2] * src[1] + mat[4];
    	    dst[1] = mat[1] * src[0] +  mat[3] * src[1] + mat[5];
    	
    }



/**
 * Transforms  arrays of double precision which represents coordinates by this transform.
 * The coordinates are stored in two differents arrays starting at zero.
 * @param srcX the array containing the source point coordinates X.
 * @param srcY the array containing the source point coordinates Y.
 * @param dstX the array into which the transformed point coordinates X are returned.
 * @param dstY the array into which the transformed point coordinates Y are returned.
 * @param numPts the number of point objects to be transformed
 */
public void transform(double[] srcX, double[] srcY, double[] dstX, double[] dstY, int numPts) {
    double[] mat = new double[6];
    afft.getMatrix(mat);
    	while (numPts-- > 0) {
    	    double x = srcX[numPts];
    	    double y = srcY[numPts];
    	    dstX[numPts] = mat[0] * x + mat[2] * y + mat[4];
    	    dstY[numPts] = mat[1] * x +  mat[3] * y + mat[5];
    	}
    }

/**
 * Transforms an array of double precision coordinates by this transform.
 * @param srcPts the array containing the source point coordinates defined in the basis of this transform.
 * Each point is stored as a pair of x,&nbsp;y coordinates.
 * @param dstPts the array into which the transformed point
 * coordinates are returned.  Each point is stored as a pair of
 * x,&nbsp;y coordinates.
 * @param srcOff the offset to the first point to be transformed
 * in the source array
 * @param dstOff the offset to the location of the first
 * transformed point that is stored in the destination array
 * @param numPts the number of point objects to be transformed
 */
public void transform(double[] srcPts, int srcOff,
    		      double[] dstPts, int dstOff,
    		      int numPts) {
afft.transform(srcPts,srcOff, dstPts,dstOff,numPts);
}


/**
 * Transforms  a point represented by a Point2D. 
 * @param Point2D, the point to be transformed.
 */
public void transform(Point2D p) {
    double[] tab = {p.getX(),p.getY()};
    double[] dest = new double[2];
    if (getOIJ().equals(p.getOIJ()))
    	{
	this.transform(tab,dest);
	p.setLocation(dest[0],dest[1]);
	}
    else
    	{
	if (Debug.On) Debug.print("No optimizations for the moment in AffineTransform2D::transform(Point2D)");
	Matrix2D mv = new Matrix2D();
	p.getOIJ().matPassage(getOIJ(),mv);
	double[] newp = mv.compute(tab[0],tab[1]);
	this.transform(newp,dest);
	getOIJ().matPassage(p.getOIJ(),mv);
	dest = mv.compute(dest[0],dest[1]);
	p.setLocation(dest[0],dest[1]);
	}
	    	
    }

public void writeMore(PrintWriter f, OpaleSet p) throws InvalidFormatException
	{
	double[] mat = new double[6];
	afft.getMatrix(mat);
	f.println("Matrix \t"+mat[0]+" "+mat[2]+ " "+mat[4]);
	f.println("\t"+mat[1]+" "+mat[3]+ " "+mat[5]);
	}	
	
public int readKeyWord(String word, StreamReader f, OpaleSet p) throws java.io.IOException, InvalidFormatException
	{
	if ( word.equals("Matrix"))
		{
		double m00, m01, m02, m10, m11, m12;
		m00 = f.nextDouble();
		m01 = f.nextDouble();
		m02 = f.nextDouble();
		m10 = f.nextDouble();
		m11 = f.nextDouble();
		m12 = f.nextDouble();
		afft.setTransform(m00,m10,m01,m11,m02,m12);
		return 0;
		}
	return -1;
		
	}
/**
* Effectue un changement de repere de la transformation.
* @param OIJ rep, le nouveau repere.
*/
public void changeOIJ(OIJ rep)
	{
	if (Debug.On) Debug.print("No optimizations for the moment in AffineTransform2D::changeOIJ");
	Matrix2D mv = new Matrix2D();
	rep.matPassage(getOIJ(),mv);
	double[][] coeff = mv.getMatrix();
	
	try
	{
	
        AffineTransform P = new AffineTransform(coeff[0][0],coeff[1][0],coeff[0][1],coeff[1][1],coeff[0][2],coeff[1][2]);
	AffineTransform invP = P.createInverse();
	afft.concatenate(P);
	invP.concatenate(afft);
	afft.setTransform(invP);
		
	setOIJ(rep);
	}
	catch(java.awt.geom.NoninvertibleTransformException evt)
		{
		System.err.println(evt);
		}
	}
	
	
/**
 * Returns <code>true</code> if this <code>AffineTransform2D</code> is
 * an identity transform in its definition basis.
 * @return <code>true</code> if this <code>AffineTransform2D</code> is
 * an identity transform; <code>false</code> otherwise.
 */
public boolean isIdentity() {
    return afft.isIdentity();
}
	
public String toString()
	{
	return afft.toString();
	}

}
