/******************************************************************************
  FontMetrics.c

  Purpose:
    This file provides access to fontmetrics, a line breaking algorithm for
	text wrapping, and string width calculations, metrics for the 14 standard
	PDF fonts are defined below
******************************************************************************/

#if defined _WINDOWS
#include "stdafx.h"
#else
#include <panda/constants.h>
#include <panda/functions.h>
#endif
#include <math.h>

/* Generated From .afm files for the 14 Core PDF fonts see:-         */
/* http://partners.adobe.com/asn/developer/technotes/acrobatpdf.html */
panda_fontmetric fontmetrics[] = {
  {"/Courier",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 0, 600, 600, 600, 600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0,
    600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0, 600, 600, 0, 600, 600,
    600, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 600, 0, 600, 0,
    0, 0, 0, 600, 600, 600, 600, 0, 0, 0, 0, 0, 600, 0, 0, 0, 600, 0, 0, 600,
    600, 600, 600, 0, 0, 0, 0}},
  {"/Courier-Bold",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 0, 600, 600, 600, 600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0,
    600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0, 600, 600, 0, 600, 600,
    600, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 600, 0, 600, 0,
    0, 0, 0, 600, 600, 600, 600, 0, 0, 0, 0, 0, 600, 0, 0, 0, 600, 0, 0, 600,
    600, 600, 600, 0, 0, 0, 0}},
  {"/Courier-BoldOblique",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 0, 600, 600, 600, 600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0,
    600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0, 600, 600, 0, 600, 600,
    600, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 600, 0, 600, 0,
    0, 0, 0, 600, 600, 600, 600, 0, 0, 0, 0, 0, 600, 0, 0, 0, 600, 0, 0, 600,
    600, 600, 600, 0, 0, 0, 0}},
  {"/Courier-Oblique",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
    600, 0, 600, 600, 600, 600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0,
    600, 0, 600, 600, 600, 600, 600, 600, 600, 600, 0, 600, 600, 0, 600, 600,
    600, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 600, 0, 600, 0,
    0, 0, 0, 600, 600, 600, 600, 0, 0, 0, 0, 0, 600, 0, 0, 0, 600, 0, 0, 600,
    600, 600, 600, 0, 0, 0, 0}},
  {"/Helvetica",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 278, 278, 355, 556, 556, 889, 667, 222, 333, 333,
    389, 584, 278, 333, 278, 278, 556, 556, 556, 556, 556, 556, 556, 556, 556,
    556, 278, 278, 584, 584, 584, 556, 1015, 667, 667, 722, 722, 667, 611,
    778, 722, 278, 500, 667, 556, 833, 722, 778, 667, 778, 722, 667, 611, 722,
    667, 944, 667, 667, 611, 278, 278, 278, 469, 556, 222, 556, 556, 500, 556,
    556, 278, 556, 556, 222, 222, 500, 222, 833, 556, 556, 556, 556, 333, 500,
    278, 556, 500, 722, 500, 500, 500, 334, 260, 334, 584, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 333, 556, 556, 167, 556, 556, 556, 556, 191, 333, 556, 333, 333,
    500, 500, 0, 556, 556, 556, 278, 0, 537, 350, 222, 333, 333, 556, 1000,
    1000, 0, 611, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0,
    333, 333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1000,
    0, 370, 0, 0, 0, 0, 556, 778, 1000, 365, 0, 0, 0, 0, 0, 889, 0, 0, 0, 278,
    0, 0, 222, 611, 944, 611, 0, 0, 0, 0}},
  {"/Helvetica-Bold",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 278, 333, 474, 556, 556, 889, 722, 278, 333, 333,
    389, 584, 278, 333, 278, 278, 556, 556, 556, 556, 556, 556, 556, 556, 556,
    556, 333, 333, 584, 584, 584, 611, 975, 722, 722, 722, 722, 667, 611, 778,
    722, 278, 556, 722, 611, 833, 722, 778, 667, 778, 722, 667, 611, 722, 667,
    944, 667, 667, 611, 333, 278, 333, 584, 556, 278, 556, 611, 556, 611, 556,
    333, 611, 611, 278, 278, 556, 278, 889, 611, 611, 611, 611, 389, 556, 333,
    611, 556, 778, 556, 556, 500, 389, 280, 389, 584, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 333, 556, 556, 167, 556, 556, 556, 556, 238, 500, 556, 333, 333, 611,
    611, 0, 556, 556, 556, 278, 0, 556, 350, 278, 500, 500, 556, 1000, 1000,
    0, 611, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0, 333,
    333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1000, 0,
    370, 0, 0, 0, 0, 611, 778, 1000, 365, 0, 0, 0, 0, 0, 889, 0, 0, 0, 278, 0,
    0, 278, 611, 944, 611, 0, 0, 0, 0}},
  {"/Helvetica-BoldOblique",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 278, 333, 474, 556, 556, 889, 722, 278, 333, 333,
    389, 584, 278, 333, 278, 278, 556, 556, 556, 556, 556, 556, 556, 556, 556,
    556, 333, 333, 584, 584, 584, 611, 975, 722, 722, 722, 722, 667, 611, 778,
    722, 278, 556, 722, 611, 833, 722, 778, 667, 778, 722, 667, 611, 722, 667,
    944, 667, 667, 611, 333, 278, 333, 584, 556, 278, 556, 611, 556, 611, 556,
    333, 611, 611, 278, 278, 556, 278, 889, 611, 611, 611, 611, 389, 556, 333,
    611, 556, 778, 556, 556, 500, 389, 280, 389, 584, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 333, 556, 556, 167, 556, 556, 556, 556, 238, 500, 556, 333, 333, 611,
    611, 0, 556, 556, 556, 278, 0, 556, 350, 278, 500, 500, 556, 1000, 1000,
    0, 611, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0, 333,
    333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1000, 0,
    370, 0, 0, 0, 0, 611, 778, 1000, 365, 0, 0, 0, 0, 0, 889, 0, 0, 0, 278, 0,
    0, 278, 611, 944, 611, 0, 0, 0, 0}},
  {"/Helvetica-Oblique",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 278, 278, 355, 556, 556, 889, 667, 222, 333, 333,
    389, 584, 278, 333, 278, 278, 556, 556, 556, 556, 556, 556, 556, 556, 556,
    556, 278, 278, 584, 584, 584, 556, 1015, 667, 667, 722, 722, 667, 611,
    778, 722, 278, 500, 667, 556, 833, 722, 778, 667, 778, 722, 667, 611, 722,
    667, 944, 667, 667, 611, 278, 278, 278, 469, 556, 222, 556, 556, 500, 556,
    556, 278, 556, 556, 222, 222, 500, 222, 833, 556, 556, 556, 556, 333, 500,
    278, 556, 500, 722, 500, 500, 500, 334, 260, 334, 584, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 333, 556, 556, 167, 556, 556, 556, 556, 191, 333, 556, 333, 333,
    500, 500, 0, 556, 556, 556, 278, 0, 537, 350, 222, 333, 333, 556, 1000,
    1000, 0, 611, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0,
    333, 333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1000,
    0, 370, 0, 0, 0, 0, 556, 778, 1000, 365, 0, 0, 0, 0, 0, 889, 0, 0, 0, 278,
    0, 0, 222, 611, 944, 611, 0, 0, 0, 0}},
  {"/Symbol",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 250, 333, 713, 500, 549, 833, 778, 439, 333, 333,
    500, 549, 250, 549, 250, 278, 500, 500, 500, 500, 500, 500, 500, 500, 500,
    500, 278, 278, 549, 549, 549, 444, 549, 722, 667, 722, 612, 611, 763, 603,
    722, 333, 631, 722, 686, 889, 722, 722, 768, 741, 556, 592, 611, 690, 439,
    768, 645, 795, 611, 333, 863, 333, 658, 500, 500, 631, 549, 549, 494, 439,
    521, 411, 603, 329, 603, 549, 549, 576, 521, 549, 549, 521, 549, 603, 439,
    576, 713, 686, 493, 686, 494, 480, 200, 480, 549, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    750, 620, 247, 549, 167, 713, 500, 753, 753, 753, 753, 1042, 987, 603,
    987, 603, 400, 549, 411, 549, 549, 713, 494, 460, 549, 549, 549, 549,
    1000, 603, 1000, 658, 823, 686, 795, 987, 768, 768, 823, 768, 768, 713,
    713, 713, 713, 713, 713, 713, 768, 713, 790, 790, 890, 823, 549, 250, 713,
    603, 603, 1042, 987, 603, 987, 603, 494, 329, 790, 790, 786, 713, 384,
    384, 384, 384, 384, 384, 494, 494, 494, 494, 0, 329, 274, 686, 686, 686,
    384, 384, 384, 384, 384, 384, 494, 494, 494}},
  {"/Times-Bold",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 250, 333, 555, 500, 500, 1000, 833, 333, 333, 333,
    500, 570, 250, 333, 250, 278, 500, 500, 500, 500, 500, 500, 500, 500, 500,
    500, 333, 333, 570, 570, 570, 500, 930, 722, 667, 722, 722, 667, 611, 778,
    778, 389, 500, 778, 667, 944, 722, 778, 611, 778, 722, 556, 667, 722, 722,
    1000, 722, 722, 667, 333, 278, 333, 581, 500, 333, 500, 556, 444, 556,
    444, 333, 500, 556, 278, 333, 556, 278, 833, 556, 500, 556, 556, 444, 389,
    333, 556, 500, 722, 500, 500, 444, 394, 220, 394, 520, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 333, 500, 500, 167, 500, 500, 500, 500, 278, 500, 500, 333, 333,
    556, 556, 0, 500, 500, 500, 250, 0, 540, 350, 333, 500, 500, 500, 1000,
    1000, 0, 500, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0,
    333, 333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1000,
    0, 300, 0, 0, 0, 0, 667, 778, 1000, 330, 0, 0, 0, 0, 0, 722, 0, 0, 0, 278,
    0, 0, 278, 500, 722, 556, 0, 0, 0, 0}},
  {"/Times-BoldItalic",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 250, 389, 555, 500, 500, 833, 778, 333, 333, 333,
    500, 570, 250, 333, 250, 278, 500, 500, 500, 500, 500, 500, 500, 500, 500,
    500, 333, 333, 570, 570, 570, 500, 832, 667, 667, 667, 722, 667, 667, 722,
    778, 389, 500, 667, 611, 889, 722, 722, 611, 722, 667, 556, 611, 722, 667,
    889, 667, 611, 611, 333, 278, 333, 570, 500, 333, 500, 500, 444, 500, 444,
    333, 500, 556, 278, 278, 500, 278, 778, 556, 500, 500, 500, 389, 389, 278,
    556, 444, 667, 500, 444, 389, 348, 220, 348, 570, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 389, 500, 500, 167, 500, 500, 500, 500, 278, 500, 500, 333, 333, 556,
    556, 0, 500, 500, 500, 250, 0, 500, 350, 333, 500, 500, 500, 1000, 1000,
    0, 500, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0, 333,
    333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 944, 0,
    266, 0, 0, 0, 0, 611, 722, 944, 300, 0, 0, 0, 0, 0, 722, 0, 0, 0, 278, 0,
    0, 278, 500, 722, 500, 0, 0, 0, 0}},
  {"/Times-Italic",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 250, 333, 420, 500, 500, 833, 778, 333, 333, 333,
    500, 675, 250, 333, 250, 278, 500, 500, 500, 500, 500, 500, 500, 500, 500,
    500, 333, 333, 675, 675, 675, 500, 920, 611, 611, 667, 722, 611, 611, 722,
    722, 333, 444, 667, 556, 833, 667, 722, 611, 722, 611, 500, 556, 722, 611,
    833, 611, 556, 556, 389, 278, 389, 422, 500, 333, 500, 500, 444, 500, 444,
    278, 500, 500, 278, 278, 444, 278, 722, 500, 500, 500, 500, 389, 389, 278,
    500, 444, 667, 444, 444, 389, 400, 275, 400, 541, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 389, 500, 500, 167, 500, 500, 500, 500, 214, 556, 500, 333, 333, 500,
    500, 0, 500, 500, 500, 250, 0, 523, 350, 333, 556, 556, 500, 889, 1000, 0,
    500, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0, 333, 333,
    333, 889, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 889, 0, 276, 0,
    0, 0, 0, 556, 722, 944, 310, 0, 0, 0, 0, 0, 667, 0, 0, 0, 278, 0, 0, 278,
    500, 667, 500, 0, 0, 0, 0}},
  {"/Times-Roman",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 250, 333, 408, 500, 500, 833, 778, 333, 333, 333,
    500, 564, 250, 333, 250, 278, 500, 500, 500, 500, 500, 500, 500, 500, 500,
    500, 278, 278, 564, 564, 564, 444, 921, 722, 667, 667, 722, 611, 556, 722,
    722, 333, 389, 722, 611, 889, 722, 722, 556, 722, 667, 556, 611, 722, 722,
    944, 722, 722, 611, 333, 278, 333, 469, 500, 333, 444, 500, 444, 500, 444,
    333, 500, 500, 278, 278, 500, 278, 778, 500, 500, 500, 500, 333, 389, 278,
    500, 500, 722, 500, 500, 444, 480, 200, 480, 541, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 333, 500, 500, 167, 500, 500, 500, 500, 180, 444, 500, 333, 333, 556,
    556, 0, 500, 500, 500, 250, 0, 453, 350, 333, 444, 444, 500, 1000, 1000,
    0, 444, 0, 333, 333, 333, 333, 333, 333, 333, 333, 0, 333, 333, 0, 333,
    333, 333, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 889, 0,
    276, 0, 0, 0, 0, 611, 722, 889, 310, 0, 0, 0, 0, 0, 667, 0, 0, 0, 278, 0,
    0, 278, 500, 722, 500, 0, 0, 0, 0}},
  {"/ZapfDingbats",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 278, 974, 961, 974, 980, 719, 789, 790, 791, 690,
    960, 939, 549, 855, 911, 933, 911, 945, 974, 755, 846, 762, 761, 571, 677,
    763, 760, 759, 754, 494, 552, 537, 577, 692, 786, 788, 788, 790, 793, 794,
    816, 823, 789, 841, 823, 833, 816, 831, 923, 744, 723, 749, 790, 792, 695,
    776, 768, 792, 759, 707, 708, 682, 701, 826, 815, 789, 789, 707, 687, 696,
    689, 786, 787, 713, 791, 785, 791, 873, 761, 762, 762, 759, 759, 892, 892,
    788, 784, 438, 138, 277, 415, 392, 392, 668, 668, 0, 390, 390, 317, 317,
    276, 276, 509, 509, 410, 410, 234, 234, 334, 334, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 732, 544, 544, 910, 667, 760, 760, 776,
    595, 694, 626, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788,
    788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788,
    788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 788, 894, 838,
    1016, 458, 748, 924, 748, 918, 927, 928, 928, 834, 873, 828, 924, 924,
    917, 930, 931, 463, 883, 836, 836, 867, 867, 696, 696, 874, 0, 874, 760,
    946, 771, 865, 771, 888, 967, 888, 831, 873, 927, 970, 918}},
  {"*",
   {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0}},
  NULL
};


/* Locate the metric for the currently selected font, if not available fall */
/* back on the empty "*" font metric (all zero widths, so no wrapping       */
panda_fontmetric *
panda_getfontmetric (panda_pdf * output)
{
  char *fontname;
  panda_object *font = NULL;
  panda_fontmetric *fontmetric = &fontmetrics[0];

  font = panda_getfontobj (output, output->currentFont);
  fontname = panda_finddictitem (output, font, "BaseFont");
  fontname = panda_dbread (output, fontname);

  while (fontmetric && fontmetric->fontName &&
	 // Dirty hack: Anyone know if Win32 includes a strcasecmp that I'm missing?
#ifdef _WINDOWS
	 (strcmp (fontmetric->fontName, fontname) != 0
	  && strcmp (fontmetric->fontName, "*") != 0)
#else
	 (strcasecmp (fontmetric->fontName, fontname) != 0
	  && strcasecmp (fontmetric->fontName, "*") != 0)
#endif _WINDOWS
	 )
    {
      fontmetric = fontmetric++;
    }

  /* can't do anything without some metrics */
  if (!fontmetric || !fontmetric->fontName)
    {
      panda_error (panda_true, "Unable to locate font metrics");
      return NULL;
    }

  return fontmetric;
}

/* Calculate the width of c using the specified fontmetric and the text settings */
double
panda_charwidth (panda_pdf * output, char c, panda_fontmetric * fontmetric,
		 int addCharSpacing)
{
  double w = 0.0;

  w =
    (double) fontmetric->characterWidth[c] *
    (double) output->currentFontSize / (double) 1000;
  if (c == ' ')
    w += (double) output->currentWordSpacing;
  if (addCharSpacing)
    w += (double) output->currentCharacterSpacing;

  return (w * output->currentHorizontalScaling);
}

/* Calculate the width of text using the specified fontmetric and the text settings */
/* if the fontmetric is specified as NULL then look them up for the current font    */
/* as specified in output                                                           */
double
panda_stringwidth (panda_pdf * output, char *text,
		   panda_fontmetric * fontmetric)
{
  double width = 0;
  double toPoints;
  int i = 0;

  /* Empty string has no length */
  if (!text)
    return 0;

  if (!fontmetric)
    {
      fontmetric = panda_getfontmetric (output);
    }

  if (!fontmetric)
    {
      return -1;
    }

  toPoints = ((double) output->currentFontSize) / 1000;

  for (i = 0; i < strlen (text); i++)
    {
      width += ((double) fontmetric->characterWidth[text[i]]) * toPoints;
      if (text[i] == ' ')
	{
	  width += output->currentWordSpacing;
	}
      else if (i > 1)
	{
	  width += output->currentCharacterSpacing;
	}
    }

  /* Adjust for Horizontal Scaling */
  width *= output->currentHorizontalScaling;

  return width;
}

/* Find the next logical line break in text for the current font settings,   */
/* assuming a maximum width of wrapWidth, if lineWidth is non NULL store the */
/* actual line width (in points) into it, return the number of character to  */
/* the next line break                                                       */
int
panda_findlinebreak (panda_pdf * output, char *text,
		     panda_fontmetric * fontmetric, int wrapwidth,
		     double *lineWidth)
{
  char *p, *lastSpace = NULL;
  double w = 0, cw, lastSpaceWidth = 0.0;

  /* Num chars is p-text */
  for (p = text; p != NULL && *p != '\0'; p++)
    {
      switch (*p)
	{

	  /* fall through if not followed by a \n (ie is aMac line */
	  /* break) Otherwise this is part of a \r\n and well get  */
	  /* it on the next iteration                              */
	case '\r':
	  if (*(p + 1) != '\n')
	    {
	      break;
	    }
	  /* If we got this far then we are at a forced wrap, what */
	  /* we have so far will fit within wrapwidth              */
	case '\n':
	  if (lineWidth != NULL)
	    {
	      *lineWidth = w;
	    }
	  return (int) (p - text + 1);
	  break;

	  /* Super/Subscript start, just eat the next character    */
	  /* since it is the value for the vertical offset         */
	case 4:
	case 5:
	  *p++;
	  break;

	  /* Normal script this would probably be ok in default:   */
	case 6:
	  break;

	  /* Fall through */
	case ' ':
	  lastSpace = p;
	  lastSpaceWidth = w;

	default:
	  cw = panda_charwidth (output, *p, fontmetric, (p > text) ? 1 : 0);
	  if (cw + w > wrapwidth)
	    {			/* At the end */
	      if (lastSpace)
		{
		  cw = panda_charwidth (output, ' ', fontmetric, 1);
		  w = lastSpaceWidth + cw;
		  while (*lastSpace == ' ' && lastSpace > text)
		    {
		      lastSpace--;
		      w -= cw;
		    }
		  if (lineWidth != NULL)
		    {
		      *lineWidth = w;
		    }
		  return (int) (lastSpace - text + 1);
		}
	      else
		{
		  if (lineWidth != NULL)
		    {
		      *lineWidth = w;
		    }
		  return (int) (p - text);
		}
	    }
	  w += cw;
	  break;

	}
    }
  if (lineWidth != NULL)
    {
      *lineWidth = w;
    }
  return (int) (p - text);
}
