/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/

package fr.gouv.culture.sdx.utils;

import fr.gouv.culture.sdx.application.Application;
import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.documentbase.LuceneDocumentBase;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.Framework;
import fr.gouv.culture.sdx.framework.FrameworkImpl;
import fr.gouv.culture.sdx.repository.Repository;
import fr.gouv.culture.sdx.search.lucene.FieldsDefinition;
import fr.gouv.culture.sdx.search.lucene.query.RemoteIndex;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.jvm.URLCoderWrapper;
import org.apache.avalon.excalibur.io.FileUtil;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.Logger;
import org.apache.cocoon.xml.XMLConsumer;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.Naming;
import java.util.Hashtable;
import java.util.Locale;

/**Useful programming utilities for SDX.
 *
 * Contains some useful static methods.
 */
public class
        Utilities {


    public static File getSystemTempDir() {
        String javaIoTmpDir = System.getProperty("java.io.tmpdir");
        if (Utilities.checkString(javaIoTmpDir))
            return new File(javaIoTmpDir);
        else
            return null;
    }

    /**Verifies if a given directory exists, if not the directory is created.
     *
     * @param dirPath           The path for the directory to verify.
     * @param logger            Logger to use for error handling
     */
    public static File checkDirectory(String dirPath, Logger logger) throws SDXException {

        //verifying the parameter
        if (!Utilities.checkString(dirPath)) throw new SDXException(logger, SDXExceptionCode.ERROR_DIRECTORY_PATH_NULL, null, null);
        //building a file to represent the directory
        File dir = new File(dirPath);
        //verifying the the file creation was ok
        if (dir == null) throw new SDXException(logger, SDXExceptionCode.ERROR_DIRECTORY_NULL, null, null);

        //performing directory checks
        if (!dir.exists()) {
            if (!dir.mkdirs()) {
                String[] args = new String[2];
                args[0] = dir.getAbsolutePath();
                throw new SDXException(logger, SDXExceptionCode.ERROR_CREATE_DIRS, args, null);
            } else
                Utilities.logInfo(logger, "\tThe Directory and any necessary parent directories\"" + dir.getAbsolutePath() + "\" were successfully created.");

        } else
            Utilities.logInfo(logger, "\tThe Directory \"" + dir.toString() + "\" exists.");

        if (!dir.isDirectory()) {
            String[] args = new String[2];
            args[0] = dir.getAbsolutePath();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NOT_DIRECTORY, args, null);
        }

        if (!dir.canRead()) {
            String[] args = new String[2];
            args[0] = dir.getAbsolutePath();
            throw new SDXException(logger, SDXExceptionCode.ERROR_DIRECTORY_NOT_READABLE, args, null);
        }

        if (!dir.canWrite()) {
            String[] args = new String[2];
            args[0] = dir.getAbsolutePath();
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_DIRECTORY_NOT_WRITABLE, args, null);
            Utilities.logWarn(logger, sdxE.getMessage(), null);
        }

        return dir;

    }

    /**Used to create a repository based upon a 'type' attribute
     *
     * @param repoConf  The configuration object made up of an 'repository' element.
     * @param manager   The component manager to pass to the repository
     * @param props     The properties object to pass to the repository
     * @param logger    The logger to pass to the repository and for error handling
     * @throws SDXException
     */
    public static Repository createRepository(Configuration repoConf, ComponentManager manager, Hashtable props, Logger logger) throws SDXException, ConfigurationException {
        //TODOException?:throw exceptions below logger may not be necessary-figure this out
        //at this point we have <sdx:repository> element
        Repository repo = null;
        try {
            //reading the 'type' attribute
            String repoType = repoConf.getAttribute(Repository.ATTRIBUTE_TYPE);
            //verifying the attribute
            Utilities.checkConfAttributeValue(Repository.ATTRIBUTE_TYPE, repoType, repoConf.getLocation());
            //building the fully qualified class name based on the type
            String repoClassName = Repository.PACKAGE_QUALNAME + repoType + Repository.CLASS_NAME_SUFFIX;
            //TODONOW:commenting the below line until we get our cvs addition of mysql working again-rbp
//            String repoClassName = PACKAGE_QUALNAME + repoType.toUpperCase() + CLASS_NAME_SUFFIX;
            //getting the class for the class name
            Class repoClass = null;
            try {
                repoClass = Class.forName(repoClassName);
            } catch (ClassNotFoundException e) {
                //logging the first failure
                String[] args = new String[2];
                args[0] = repoConf.getAttribute(Repository.ATTRIBUTE_ID);
                args[1] = e.getMessage();
                //logging first exception
                SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_REPOSITORY, args, e);
                Utilities.logWarn(logger, sdxE.getMessage(), null);
                //trying the attribute value, hopefully the user is providing a fully qualified attribute name
                repoClassName = repoType;
                try {
                    repoClass = Class.forName(repoClassName);
                } catch (ClassNotFoundException e1) {
                    String[] args2 = new String[2];
                    args2[0] = repoConf.getAttribute(Repository.ATTRIBUTE_ID);
                    args2[1] = e1.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_REPOSITORY, args2, e1);
                }
            }
            //building a new instance of the class
            Object obj = repoClass.newInstance();
            if (obj == null) {
                String[] args = new String[1];
                args[0] = repoClassName;
                throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_OBJECT_INSTANCE_NULL, args, null);
            }

            if (!(obj instanceof Repository)) {
                //the object doesn't implement our interface
                String[] args = new String[3];
                args[0] = "Repository";
                args[1] = repoClass.getName();
                args[2] = repoType;
                throw new SDXException(logger, SDXExceptionCode.ERROR_CLASS_NOT_INSTANCEOF_SDX_INTERFACE, args, null);
            }
            //casting into a repository object
            repo = (Repository) obj;
            //setting the logger
            repo.enableLogging(logger);
            //passing cocoon's component manager to the repo
            repo.compose(manager);
            //passing the properties object to the repo
            repo.setProperties(new Hashtable(props));
            //configuring the repo
            repo.configure(repoConf);
            //initializing the repo
            repo.init();
        } catch (InstantiationException e) {
            String[] args = new String[2];
            args[0] = repoConf.getAttribute(Repository.ATTRIBUTE_ID);
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_REPOSITORY, args, e);
        } catch (IllegalAccessException e) {
            String[] args = new String[2];
            args[0] = repoConf.getAttribute(Repository.ATTRIBUTE_ID);
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_REPOSITORY, args, e);
        } catch (ComponentException e) {
            String[] args = new String[2];
            args[0] = repoConf.getAttribute(Repository.ATTRIBUTE_ID);
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_REPOSITORY, args, e);
        }
        return repo;
    }

    /**Verifies the logger and logs a message
     *
     * @param logger    The logger
     * @param s         The message
     */
    public static void logInfo(Logger logger, String s) {
        if (logger != null && s != null)
            logger.info(s);
        else {//TODO?:what else can we do besides go system out
        }
    }

    /**Verifies the logger and logs a message
     *
     * @param logger    The logger
     * @param s         The message (maybe null)
     * @param e         The exception (maybe null)
     */
    public static void logDebug(Logger logger, String s, Exception e) {
        if (logger != null) {
            if (s != null) {
                if (e != null) {
                    logger.debug(s, e);
                } else
                    logger.debug(s);
            } else if (e != null && (e.getMessage() != null & e.fillInStackTrace() != null)) {
                logger.debug(e.getMessage(), e.fillInStackTrace());
            }
        } else {//TODO?:what else can we do besides go system out
        }
    }

    /**Verifies the logger and logs a message
     *
     * @param logger    The logger
     * @param s         The message (maybe null)
     * @param e         The exception (maybe null)
     */
    public static void logError(Logger logger, String s, Exception e) {
        if (logger != null) {
            if (s != null) {
                if (e != null) {
                    logger.error(s, e);
                } else
                    logger.error(s);
            } else if (e != null && (e.getMessage() != null & e.fillInStackTrace() != null)) {
                logger.error(e.getMessage(), e.fillInStackTrace());
            }
        } else {//TODO?:what else can we do besides go system out
        }
    }

    /**Verifies the logger and logs a message
     *
     * @param logger    The logger
     * @param s         The message (maybe null)
     * @param e         The exception (maybe null)
     */
    public static void logWarn(Logger logger, String s, Exception e) {
        if (logger != null) {
            if (s != null) {
                if (e != null) {
                    logger.warn(s, e);
                } else
                    logger.warn(s);
            } else if (e != null)
                logger.warn(e.getMessage(), e);

        } else {//TODO?:what else can we do besides go system out
        }
    }

    /**Verifies the logger and logs an exception
     *
     * @param logger    The logger
     * @param e         The exception (maybe null)
     */
    public static void logException(Logger logger, Exception e) {
        if (logger != null && e != null && (e.getMessage() != null && e.fillInStackTrace() != null))
            logger.error(e.getMessage(), e.fillInStackTrace());
        else {//TODO?:what else can we do besides go system out
        }
    }

    /**Checks a document to ensure the object is not null and that it's id is not null or an empty string
     *
     * @param logger    The logger for error handling
     * @param doc       The document object to verify
     * @throws SDXException Thrown if the object is null, or its id is <code>null</code> or an empty String
     */
    public static void checkDocument(Logger logger, Document doc) throws SDXException {
        //ensuring we have a valid object
        if (doc == null)
            throw new SDXException(logger, SDXExceptionCode.ERROR_DOC_NULL, null, null);
        //ensuring we have a valid unique id
        if (!Utilities.checkString(doc.getId())) {
            String[] args = new String[1];
            if (doc.getURL() != null) args[0] = doc.getURL().toString();
            throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_DOC_ID, args, null);
        }

    }

    /**Verifies that an OutputStream is not null
     *
     * @param logger    The logger for error handling
     * @param os        The OutputStream to verify
     * @throws SDXException
     */
    public static void checkOutputStream(Logger logger, OutputStream os) throws SDXException {
        if (os == null)
            throw new SDXException(logger, SDXExceptionCode.ERROR_OUTPUT_STREAM_NULL, null, null);
    }

    /**Verifies that an XMLConsumer is not null
     *
     * @param logger    The logger for error handling
     * @param consumer  The XMLConsumer to verify
     * @throws SDXException
     */
    public static void checkXmlConsumer(Logger logger, XMLConsumer consumer) throws SDXException {
        if (consumer == null)
            throw new SDXException(logger, SDXExceptionCode.ERROR_XML_CONSUMER_NULL, null, null);
    }

    /**Verifies a configuration object
     *
     * @param configuration The object to verify
     * @throws ConfigurationException   Thrown if the object is null
     */
    public static void checkConfiguration(Configuration configuration) throws ConfigurationException {
        if (configuration == null) {
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_CONFIG_NULL, null, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }
    }

    /**Verifies a configuration attribute value, ie it should not be null or an empty String
     *
     * @param attName       The name of the attribute
     * @param attValue      The value of the attribute
     * @param attLocation   The location of the configuration object
     * @throws ConfigurationException   Thrown if <code>null</code> or an empty String
     */
    public static void checkConfAttributeValue(String attName, String attValue, String attLocation) throws ConfigurationException {
        /*a null check is done for safety, but a null string value should not be passed
        *to this method as the message given will not be accurate, but will still be helpful*/
        if (!Utilities.checkString(attValue)) {
            String[] args = new String[2];
            args[0] = attName;
            args[1] = attLocation;
            //null logger to avoid double logging
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_INVALID_ATTRIBUTE, args, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }
    }

    /**This method will attempt to verify if a file exists
     *
     * @param logger            The logger if needed
     * @param confLocation      The location string from a configuration object, this way we can determine our current directory location, if needed
     * @param props             The application properties object, containing the path needed to find the file
     * @param path              The path of the document, absolute or relative to see below:
     * @param isConfDirectory       Indicates whether the file trying to be resolved could be a non-existent FSRepository base directory,
     *                          if so we would like to create it
     * <p>This method will attempt to do this in the following order:
     * <p>1)build a file object using the absolute file path, "file:/..."
     * <p>2)a file object using the base path of the web application directory and the relative path provided, must begin with a "/"
     * <p>3)a file object using the base path of the current directory (sdx configuration directory or the application configuration directory)
     *      and the relative path provided, could be something like: "../dir1/dir2/file.extension" or "dir1/file.extension", etc.
     * @return  A File object for the file if we are able to build a correct path to the existing file
     */
    public static File resolveFile(Logger logger, String confLocation, Hashtable props, String path, boolean isConfDirectory) throws SDXException {

        String currentDir = confLocation;

        if (!Utilities.checkString(path))
            throw new SDXException(null, SDXExceptionCode.ERROR_RESOLVE_FILE_INVALID_FILE_PATH, null, null);
        File file = null;
        //trying the absolute path option first
        //TODO?:should we support file over a http url in the future?-rbp
        try {
            if (path.startsWith("file:/")) file = FileUtil.toFile(new URL(path));
        } catch (MalformedURLException e) {
            String[] args = new String[1];
            args[0] = path;
            throw new SDXException(logger, SDXExceptionCode.ERROR_BUILD_URL, args, e);
        }
        if (file != null && file.exists())
            return file;
        else if (file != null && isConfDirectory)
            return Utilities.checkDirectory(file.getAbsolutePath(), logger);
        else {
            //at this point we need this object to build paths, and we need to ensure it is valid
            if (props != null) {
                /*
                String[] args = new String[1];
                args[0] = path;
                throw new SDXException(null, SDXExceptionCode.ERROR_RESOLVE_FILE_PROPS_NULL, args, null);
                 */
                //}
                //determining the current directory based upon a string check
                /*this is a hack, but i see no other way to determine our current
                *directory as our configuration architecture needs to remain generic
                *to be useful in other places*/
                try {
                    //System.out.println("currentDir before : " + currentDir);

                    if (confLocation.indexOf(FrameworkImpl.CONFIGURATION_FILE_NAME) > -1 ||
                            confLocation.indexOf(new File(Utilities.getStringFromHashtable(FrameworkImpl.SDX_CONF_PATH, props)).toURL().toExternalForm()) > -1)
                    //trying to build a relative path from the sdx configuration directory
                        currentDir = Utilities.getStringFromHashtable(FrameworkImpl.SDX_CONF_PATH, props);
                    else if (confLocation.indexOf(FrameworkImpl.APP_CONFIG_FILENAME) > -1 ||
                            confLocation.indexOf(new File(Utilities.getStringFromHashtable(FrameworkImpl.APP_CONF_PATH, props)).toURL().toExternalForm()) > -1)
                    //trying to build a relative path from the application's configuration directory
                        currentDir = Utilities.getStringFromHashtable(FrameworkImpl.APP_CONF_PATH, props);
                    else {
                        String tmp = currentDir;
                        int endIdx = tmp.lastIndexOf("/");
                        if (endIdx < 0) endIdx = tmp.length();
                        currentDir = (FileUtil.toFile(new URL(tmp.substring(0, endIdx)))).getCanonicalPath();
                    }    //System.out.println("currentDir after : " + currentDir);
                } catch (IOException e) {
                    //if not we can't resolve the file, sorry
                    //TODOException
                    String[] args = new String[1];
                    args[0] = path;
                    throw new SDXException(null, SDXExceptionCode.ERROR_RESOLVE_FILE, args, e);
                }
                //trying to build the path from the sdx application directory under "webapps"
                if (path.startsWith("/"))
                //building the path from the sdx application (cocoon app) directory and the relative path minus the intial forward slash
                    file = new File(Utilities.getStringFromHashtable(FrameworkImpl.SDX_APP_PATH, props), path.substring(1));
                else
                    file = FileUtil.resolveFile(new File(currentDir), path);
            }
        }
        if (file != null && file.exists())
            return file;
        else if (file != null && isConfDirectory)
            return Utilities.checkDirectory(file.getAbsolutePath(), logger);
        else {
            //if not we can't resolve the file, sorry
            String[] args = new String[1];
            args[0] = path;
            throw new SDXException(null, SDXExceptionCode.ERROR_RESOLVE_FILE, args, null);
        }

    }

    /**Returns true if a string is not null or empty
     *
     * @param s The string to verify
     */
    public static boolean checkString(String s) {
        if (s != null && !s.trim().equals(""))
            return true;
        else
            return false;
    }


    /**Returns a locale from a configuration object
     *
     * @param conf  The configuration object for the element which contains the "xml:lang"
     *              attribute with an optional "variant" attribute.
     * @param defaultLocale A default locale to use if the building of the local fails
     *                     if a <code>null</code> defaultLocale is passed, the system default is used.
     */
    public static Locale buildLocale(Configuration conf, Locale defaultLocale) {
        //getting 'xml:lang' attr, default is null if it doesn't exist
        String xmlLang = conf.getAttribute(Node.Name.XML_LANG, null);
        //getting 'variant' attr, default is null if it doesn't exist
        String variant = conf.getAttribute(Node.Name.VARIANT, null);
        return Utilities.buildLocale(xmlLang, variant, defaultLocale);
    }

    /**Returns a locale from a String
     *
     * @param xmlLang  A valid xml:lang attribute value.
     * @param variant  A variant from the Java specs,can be <code>null</code>
     * @param defaultLocale A default locale to use if the building of the local fails
     *                     if a <code>null</code> defaultLocale is passed, the system default is used.
     */
    public static Locale buildLocale(String xmlLang, String variant, Locale defaultLocale) {
        Locale locale = null;
        //the country portion of the xml:lang attr
        String country = "";
        //the language portion of the xml:lang attr
        String lang = "";

        //if a default was not provided we use the system default
        if (defaultLocale == null)
            defaultLocale = Locale.getDefault();

        if (Utilities.checkString(xmlLang)) {
            //hypen, "-", is the separator between the primary tag and subtag in RFC 1766
            int hypenIndex = xmlLang.indexOf("-");
            //if there is a hypen we split the until the hypen for the lang and after the hypen for the country
            if (hypenIndex > 0) {
                lang = xmlLang.substring(0, hypenIndex);
                country = xmlLang.substring(hypenIndex + 1);
            } else//if no hypen, we have been given a country only
                lang = xmlLang;
        }

        if (Utilities.checkString(lang)) {
            if (Utilities.checkString(country)) {
                if (Utilities.checkString(variant)) {
                    locale = new Locale(lang, country, variant);
                } else
                    locale = new Locale(lang, country);
            } else
                locale = new Locale(lang, "");
        } else
            locale = defaultLocale;

        return locale;
    }

    /** Build an id for an attached document from :
     *      the parent document id (@refId)
     *      a name unique relatively to the parent document (@relId)
     *  Used to store and attach files to a parent document (example: images)
     *
     * @param   baseId  the id of parent document
     * @param   relId  an id relative to the parent document
     * @return  an id surely unique in scope of a document base
     */
    public static String attId(String baseId, String relId) {
        return baseId + "_" + relId;
    }

    /** Build an URL for an attached document from :
     *      the parent document URL (@refURL)
     *      a path relative to the parent document (@path)
     *  Used for relative URL like <img href="images/myimage.jpg"/> in an HTML document
     *
     * @param   refURL  a well-formed URL
     * @param   path  a path string it could be relative or absolute
     * @return  a URL for a
     */
    public static URL attUrl(URL refURL, String path) throws MalformedURLException {
        //building document attUrl from parent document attUrl and relative attUrl
        String attUrl = path;
        try {
            //trying to build a url from an absolute path
            return new URL(path);
        } catch (MalformedURLException e) {
            //trying to build a url from a relative path
            if (refURL != null && Utilities.checkString(refURL.toExternalForm())) {
                /*
                String basePath = refURL.toExternalForm().substring(0, refURL.toString().lastIndexOf("/") + 1);
                //now we have the present directory path of the parent file
                //we make sure we have a good base attUrl (ie it ends with a forward slash)
                if (!basePath.endsWith("/")) basePath = basePath + "/";
                attUrl = basePath + path;
                */
                try {
                    //return new URL(attUrl);
                    File f = FileUtil.toFile(refURL);
                    URL resultURL = null;
                    if (f != null){
						resultURL =  FileUtil.resolveFile(FileUtil.toFile(refURL).getParentFile(), attUrl).toURL();
                    }
					else if (f == null){
						URL baseURL = new URL (refURL.toString().substring(0,refURL.toString().lastIndexOf("/")+1));
						resultURL = new URL(baseURL, attUrl);
					}
					return resultURL;
                } catch (MalformedURLException e1) {
                    try {
                        /*if all is fails we will just add the paths together, but they must be in the proper format
                        for java's URL, sorry can't do much else here*/
                        return new URL(refURL, attUrl);
                    } catch (MalformedURLException e2) {
                        throw new MalformedURLException(e.getMessage() + "\n" + e1.getMessage() + "\n" + e2.getMessage());
                    }
                }
            } else
                throw e;
        }

    }


    /**Return's a string value from a hashtable by making the appropriate cast
     *
     * @param key   The key for the string value
     * @param table The hashtable
     * @return
     */
    public static String getStringFromHashtable(String key, Hashtable table) {
        String value = null;
        if (Utilities.checkString(key) && table != null) {
            value = (String) table.get(key);
        }
        return value;
    }

    /**Builds a rmi url string for RemoteIndex lookups
     *
     * @param rmiHost   The host name or ip address of the machine
     * @param rmiPort   The port number for the rmi registry
     * @param appId     The id of the application to which the RemoteIndex belongs
     * @param dbId      The id of the document base to which the RemoteIndex belongs
     * @return
     */
    public static String buildRmiName(String rmiHost, int rmiPort, String appId, String dbId) {
        String rmiName = null;
        if (Utilities.checkString(rmiHost) && Utilities.checkString(appId) && Utilities.checkString(dbId))
            rmiName = "//" + rmiHost + ":" + Integer.toString(rmiPort) + "/" + appId + "_" + dbId;
        return rmiName;
    }

    public static RemoteIndex getRemoteIndex(Logger logger, String remoteIndexName) throws SDXException {
        RemoteIndex rIndex = null;
        try {
            rIndex = (RemoteIndex) Naming.lookup(remoteIndexName);

        } catch (Exception e) {
            String[] args = new String[1];
            args[0] = remoteIndexName;
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_REMOTE_INDEX, args, e);
        }
        return rIndex;
    }

    /**Joins an array of strings
     *
     * @param strings   The array of strings
     * @param delimiter The delimiter, if none desired use <code>null</code>
     * @return  The joined string, or <code>null</code> if the array was null or empty
     */
    public static String joinStrings(String[] strings, String delimiter) {
        if (strings == null || strings.length == 0) return null;

        if (delimiter == null) delimiter = "";
        String newString = null;
        for (int i = 0; i < strings.length; i++) {

            if (strings[i] == null) strings[i] = "";
            //if it is the first string, we dont want to add the delimiter
            if (i == 0)
                newString = strings[i];
            else
                newString = newString + delimiter + strings[i];
        }
        return newString;
    }

    public static Application getApplication(ComponentManager manager, Hashtable props) throws SDXException {
        Application app = null;
        String appId = Utilities.getStringFromHashtable(Application.APPLICATION_ID, props);
        if (Utilities.checkString(appId)) {
            try {
                FrameworkImpl frame = (FrameworkImpl) manager.lookup(Framework.ROLE);
                if (frame != null)
                    app = frame.getApplicationById(appId);
            } catch (ComponentException e) {
            }

        }
        return app;
    }

    public static String encodeURL(String url, String encoding) {
        try {
            return URLCoderWrapper.encode(url, encoding);
        } catch (Exception e) {
            try {
                return URLCoderWrapper.encode(url, Framework.DEFAULT_ENCODING);
            } catch (Exception e1) {
                return url;
            }
        }
    }

    public static String decodeURL(String url, String encoding) {
        try {
            return URLCoderWrapper.decode(url, encoding);
        } catch (Exception e) {
            try {
                return URLCoderWrapper.decode(url, Framework.DEFAULT_ENCODING);
            } catch (Exception e1) {
                return url;
            }
        }
    }

    public static FieldsDefinition configureFieldList(Logger logger, Configuration configuration, Hashtable props) throws ConfigurationException {
        //Lucene specific configuration
        FieldsDefinition fieldsDef = null;
        //at this point, we should have <sdx:fieldList> containing a list of fields
        try {
            //getting the list of fields for indexing
            Configuration fieldList = configuration.getChild(LuceneDocumentBase.ELEMENT_NAME_FIELD_LIST, false);
            //testing if we have something
            if (fieldList == null) {
                String[] args = new String[1];
                args[0] = configuration.getLocation();
                SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_FIELD_LIST_CONFIG, args, null);
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }

            //assigning a value for the class field
            fieldsDef = new FieldsDefinition();
            //setting the logger
            fieldsDef.enableLogging(logger);
            //setting properties
            fieldsDef.setProperties(props);
            //configuring the fieldsDefinition with the fields list
            fieldsDef.configure(fieldList);

            if (props != null) {
                //we add SDX internal fields
                Configuration sdxConf = (Configuration) props.get(FrameworkImpl.SDX_CONF);
                //testing if we have something
                if (sdxConf == null) {
                    String[] args = new String[2];
                    args[0] = FrameworkImpl.CONFIGURATION_FILE_NAME;
                    args[1] = Utilities.getStringFromHashtable(FrameworkImpl.SDX_CONF_PATH, props);
                    SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_SDX_CONFIG_FILE, args, null);
                    throw new ConfigurationException(sdxE.getMessage(), sdxE);
                }
                Configuration iFConf = sdxConf.getChild(LuceneDocumentBase.ELEMENT_NAME_LUCENE_SDX_INTERNAL_FIELDS, false);
                //testing if we have something
                if (iFConf == null) {
                    //TODOException?:should we be warning here instead of throwing an exception?-rbp
                    String[] args = new String[1];
                    args[0] = sdxConf.getLocation();
                    SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_INTERNAL_FIELDS_CONFIG, args, null);
                    throw new ConfigurationException(sdxE.getMessage(), sdxE);
                }
                //configuring internal field for term searching
                fieldsDef.addInternalFields(iFConf);
            }

        } catch (SDXException e) {
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        }

        return fieldsDef;
    }

    public static String prefixNodeNameSDX(String elemName){
        if (!Utilities.checkString(elemName)) return null;
        String qualifiedElementName = Framework.SDXNamespacePrefix + ":" + elemName;
        return qualifiedElementName;
    }


}
