/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
 */
package fr.gouv.culture.sdx.oai;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import org.apache.avalon.excalibur.io.FileUtil;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.serialization.XMLSerializer;
import org.apache.cocoon.xml.IncludeXMLConsumer;
import org.apache.cocoon.xml.XMLPipe;
import org.apache.excalibur.xml.sax.SAXParser;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import fr.gouv.culture.oai.AbstractOAIHarvester;
import fr.gouv.culture.oai.OAIObject;
import fr.gouv.culture.oai.OAIRequest;
import fr.gouv.culture.oai.OAIRequestImpl;
import fr.gouv.culture.oai.util.OAIUtilities;
import fr.gouv.culture.sdx.document.IndexableDocument;
import fr.gouv.culture.sdx.document.OAIDocument;
import fr.gouv.culture.sdx.document.XMLDocument;
import fr.gouv.culture.sdx.documentbase.DocumentBase;
import fr.gouv.culture.sdx.documentbase.IDGenerator;
import fr.gouv.culture.sdx.documentbase.IndexParameters;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.Framework;
import fr.gouv.culture.sdx.pipeline.GenericPipeline;
import fr.gouv.culture.sdx.pipeline.Pipeline;
import fr.gouv.culture.sdx.repository.Repository;
import fr.gouv.culture.sdx.search.lucene.query.ComplexQuery;
import fr.gouv.culture.sdx.search.lucene.query.DateIntervalQuery;
import fr.gouv.culture.sdx.search.lucene.query.FieldQuery;
import fr.gouv.culture.sdx.search.lucene.query.SearchLocations;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.configuration.ConfigurationUtils;
import fr.gouv.culture.sdx.utils.constants.ContextKeys;
import fr.gouv.culture.sdx.utils.database.Database;
import fr.gouv.culture.sdx.utils.database.DatabaseBacked;
import fr.gouv.culture.sdx.utils.database.DatabaseEntity;
import fr.gouv.culture.sdx.utils.database.Property;
import fr.gouv.culture.sdx.utils.logging.LoggingUtils;
import fr.gouv.culture.sdx.utils.save.SaveParameters;
import fr.gouv.culture.sdx.utils.save.Saveable;
import fr.gouv.culture.util.apache.avalon.cornerstone.services.scheduler.SimpleTimeScheduler;
import fr.gouv.culture.util.apache.avalon.cornerstone.services.scheduler.TimeScheduler;
import fr.gouv.culture.util.apache.avalon.cornerstone.services.scheduler.TimeTrigger;
import fr.gouv.culture.util.apache.avalon.cornerstone.services.scheduler.TimeTriggerFactory;

public abstract class AbstractDocumentBaseOAIHarvester extends AbstractOAIHarvester implements DocumentBaseOAIHarvester {

	/**The underlying document base*/
	protected DocumentBase docbase = null;
	/**Id of the underlying document base*/
	protected String docbaseId = "";
	/**Pre-indexation pipeline*/
	protected Pipeline pipe = null;
	/**Underlying database to store any info*/
	protected Database _database = null;
	/**Requests in application.xconf*/
	protected Hashtable storedRequests = null;
	/**References to the underlying documentbase's/application's repositories*/
	protected Hashtable storeRepositoriesRefs = null;
	/**Time scheduler for stored requests*/
	protected TimeScheduler scheduler = null;
	/**IDGenerator for this object*/
	protected IDGenerator harvesterIdGen = null;

	//variables for sax stream handling
	protected String TEMPFILE_SUFFIX = ".sdx";
	protected File tempDir = null;
	protected File harvestDoc = null;
	protected FileOutputStream fileOs = null;
	protected XMLDocument urlResource = null;
	protected ArrayList deletedDocs = null;
	protected int noHarvestedDocs = 0;
	protected int noDocsDeleted = 0;

	protected boolean keepDeletedRecords = false;//Defaulted, we will delete "deletedRecords" keeping our harvester in sync with source repository
	protected int noRecordsPerBatch = OAIObject.NUMBER_RECORDS_PER_RESPONSE;

	/**XML Transformer factory classe name.
	 *<p>Default: Xalan, "org.apache.xalan.processor.TransformerFactoryImpl".
	 *This cas be change in configuration file:
	 *&lt;oai-harvester transformer-factory="{classe name}" [...]&gt;</p>
	 */
	protected String transformerFactory = "";
	protected String defaultTransformerFactory = "com.icl.saxon.TransformerFactoryImpl"; // Saxon 6.x
	//protected String defaultTransformerFactory = "net.sf.saxon.TransformerFactoryImpl"; // Saxon 8.x and higher
	//protected String defaultTransformerFactory = "org.apache.xalan.processor.TransformerFactoryImpl"; // Xalan
	/**XML Transformer indent option.
	 *<p>Default:no. This can be change in configuration file:
	 *&lt;oai-harvester transformer-indent="yes|no" [...]&gt;</p>
	 */
	protected String transformerIndent = "";
	protected String defaultTransformerIndent = "no";

	//strings for internal database properties and sax output
	protected static final String OAI_HARVEST_ID = "oaiHarvestId";
	// strings for XML transformer factory classe name
	protected static final String TRANSFORMER_FACTORY = "transformer-factory";
	// string for XML transformer indent option
	protected static final String TRANSFORMER_INDENT = "transformer-indent";
	//internal database property for failed harvests caused by internal errors( errors in this implementation and not OAI repository errors)
	protected static final String OAI_FAILED_HARVEST = "oaiFailedHarvest";
	protected static final String OAI_HARVESTER_LAST_UPDATED = "oaiHarvesterLastUpdated";
	protected static final String OAI_HARVESTER_RESUMPTION_TOKEN = "oaiHarvesterResumptionToken";
	/*Field names for common request parameters*/
	protected static final String OAI_VERB = "oaiVerb";
	protected static final String OAI_IDENTIFIER = "oaiIdentifier";
	protected static final String OAI_METADATA_PREFIX = "oaiMetadataPrefix";
	protected static final String OAI_FROM = "oaiFrom";
	protected static final String OAI_UNTIL = "oaiUntil";
	protected static final String OAI_SET = "oaiSet";
	protected static final String NO_DOCS_DELETED = "noDocDeleted";
	protected static final String NO_DOCS_HARVESTED = "noDocHarvested";

	/**List OAI files with OAI properties*/
	protected Hashtable filesProperties = null;

	protected XMLSerializer cBytes = null;
	protected XMLPipe oaiStripper = null;


	/**Basic constructor*/
	public AbstractDocumentBaseOAIHarvester(DocumentBase base) {
		this.docbase = base;
		if (this.docbase != null)
			this.docbaseId = this.docbase.getId();
	}


	/**Configures this object*/
	public void configure(Configuration configuration) throws ConfigurationException {
		super.userAgent = configuration.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.USER_AGENT, "SDX OAI Harvester");
		this.keepDeletedRecords = configuration.getAttributeAsBoolean(DocumentBaseOAIHarvester.ConfigurationNode.KEEP_DELETED_RECORDS, this.keepDeletedRecords);
		this.noRecordsPerBatch = configuration.getAttributeAsInteger(DocumentBaseOAIHarvester.ConfigurationNode.NO_RECORDS_PER_BATCH, this.noRecordsPerBatch);
		this.transformerFactory = configuration.getAttribute(TRANSFORMER_FACTORY, this.defaultTransformerFactory);
		this.transformerIndent = configuration.getAttribute(TRANSFORMER_INDENT, this.defaultTransformerIndent);
		configureAdminEmails(configuration);
		this.configureDataProviders(configuration);
		this.configurePipeline(configuration);
		configureDatabase(configuration);
		configureHarvestIDGenerator(configuration);
	}


	/**Configures the internal database*/
	protected void configureDatabase(Configuration configuration) throws ConfigurationException {
		DatabaseBacked internalDb = new DatabaseBacked();
		try {
			internalDb.enableLogging(this.logger);
			internalDb.contextualize(Utilities.createNewReadOnlyContext(getContext()));
			internalDb.service(this.manager);
			internalDb.setId(getHarvesterId());
			internalDb.configure(configuration);
			internalDb.init();
			this._database = internalDb.getDatabase();
		} catch (SDXException e) {
			throw new ConfigurationException(e.getMessage(), e);
		} catch (ServiceException e) {
			throw new ConfigurationException(e.getMessage(), e);
		} catch (ContextException e) {
			throw new ConfigurationException(e.getMessage(), e);
		}
	}

	/**Configures the id generator for harvests*/
	protected void configureHarvestIDGenerator(Configuration configuration) throws ConfigurationException {
		this.harvesterIdGen = ConfigurationUtils.configureIDGenerator(this.logger, configuration);
		this.harvesterIdGen.setDatabase(this._database);
	}


	/**Returns an id for this harvester based upon the underlying document base id*/
	protected String getHarvesterId() {
		String hid = "";
		hid += Framework.SDXNamespacePrefix + "_";
		hid += OAIObject.Node.Prefix.OAI + "_";
		hid += "harvester" + "_";
		hid += this.docbaseId;
		return hid;
	}

	/**Configures a list of admin emails
	 * can be sub-elements, a single attribute,
	 * or both
	 *
	 * @param configuration
	 * @throws ConfigurationException
	 */
	protected void configureAdminEmails(Configuration configuration) throws ConfigurationException {
		//configure the admin email
		ArrayList locAdminEmailsList = new ArrayList();
		String firstAdminEmail = configuration.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.ADMIN_EMAIL, null);
		Configuration[] locAdminEmails = configuration.getChildren(DocumentBaseOAIHarvester.ConfigurationNode.ADMIN_EMAIL);
		if (Utilities.checkString(firstAdminEmail))
			locAdminEmailsList.add(firstAdminEmail);
		for (int i = 0; i < locAdminEmails.length; i++) {
			Configuration locAdminEmail = locAdminEmails[i];
			if (locAdminEmail != null) {
				String value = locAdminEmail.getValue();
				if (Utilities.checkString(value))
					locAdminEmailsList.add(value);
			}
		}
		if (locAdminEmailsList.size() <= 0)//no admin email throw an error TODO:make this exception better
		ConfigurationUtils.checkConfAttributeValue(DocumentBaseOAIHarvester.ConfigurationNode.ADMIN_EMAIL, null, configuration.getLocation());

		super.adminEmails = (String[]) locAdminEmailsList.toArray(new String[0]);
		//releasing resources
		locAdminEmailsList.clear();
		locAdminEmailsList = null;

	}

	/**Configures data providers info that can be reused
	 * and from which requests can be automatically executed
	 *
	 * @param configuration
	 * @throws ConfigurationException
	 * @see #storedRequests
	 */
	protected void configureDataProviders(Configuration configuration) throws ConfigurationException {
		if (configuration != null) {
			Configuration dataProvidersConf = configuration.getChild(DocumentBaseOAIHarvester.ConfigurationNode.OAI_DATA_PROVIDERS, false);
			if (dataProvidersConf != null) {
				Configuration[] repoRequestConfs = dataProvidersConf.getChildren(DocumentBase.ConfigurationNode.OAI_REPOSITORY);
				if (repoRequestConfs != null) {
					for (int x = 0; x < repoRequestConfs.length; x++) {
						Configuration repoRequestConf = repoRequestConfs[x];
						if (repoRequestConf != null) {
							if (storedRequests == null) storedRequests = new Hashtable();
							String repoUrl = repoRequestConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.URL);//On pourrait controler l'URL en faisant un verb Identify -mp
							String repoGranularity = repoRequestConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.GRANULARITY, OAIObject.Node.Value.STRING_GRANULARITY_SECOND);//Configure repo granularity. Default is seconds "yyyy-MM-dd'T'HH:mm:ss'Z'"
							checkGranularity(repoGranularity);
							configureStoreRepositories(repoUrl, repoRequestConf);
							Configuration updateConf = repoRequestConf.getChild(DocumentBaseOAIHarvester.ConfigurationNode.UPDATE, false);
							Configuration[] verbConfs = repoRequestConf.getChildren(DocumentBaseOAIHarvester.ConfigurationNode.OAI_VERB);
							if (verbConfs != null) {
								for (int y = 0; y < verbConfs.length; y++) {
									try {
										OAIRequest request = null;
										Configuration verbConf = verbConfs[y];
										if (verbConf != null) {
											String verb = verbConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.NAME);
											String mdPrefix = verbConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.METADATA_PREFIX);
											ConfigurationUtils.checkConfAttributeValue(DocumentBaseOAIHarvester.ConfigurationNode.METADATA_PREFIX, mdPrefix, verbConf.getLocation());
											String verbId = verbConf.getAttribute("id");
											ConfigurationUtils.checkConfAttributeValue("id", verbId, verbConf.getLocation());
											if (verb.equalsIgnoreCase(OAIRequest.VERB_STRING_GET_RECORD)) {
												verb = OAIRequest.VERB_STRING_GET_RECORD;
												Configuration[] idsConf = verbConf.getChildren(DocumentBaseOAIHarvester.ConfigurationNode.OAI_IDENTIFIER);
												if (idsConf != null) {
													for (int z = 0; z < idsConf.length; z++) {
														Configuration idConf = idsConf[z];
														if (idConf != null) {
															String id = idConf.getValue();
															request = new OAIRequestImpl();
															request.enableLogging(this.logger);
															request.setRepositoryURL(repoUrl);
															request.setGranularity(repoGranularity);
															request.setVerbString(verb);
															request.setMetadataPrefix(mdPrefix);
															request.setVerbId(verbId);
															request.setIdentifier(id);
															Utilities.isObjectUnique(storedRequests, request.getRequestURL(), request);
															storedRequests.put(request.getRequestURL(), request);
															configureUpdateTriggers(request.getRequestURL(), updateConf);
														}

													}
												}

											} else if (verb.equalsIgnoreCase(OAIRequest.VERB_STRING_LIST_RECORDS)) {
												verb = OAIRequest.VERB_STRING_LIST_RECORDS;
												String from = verbConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.FROM, null);
												String until = verbConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.UNTIL, null);
												String set = verbConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.SET, null);
												boolean useLastHarvestDate = verbConf.getAttributeAsBoolean(DocumentBaseOAIHarvester.ConfigurationNode.ATTRIBUTE_USE_LAST_HARVEST_DATE, /*true by default*/true);
												request = new OAIRequestImpl();
												request.enableLogging(this.logger);
												request.setRepositoryURL(repoUrl);
												request.setGranularity(repoGranularity);
												request.setVerbString(verb);
												request.setMetadataPrefix(mdPrefix);
												request.setVerbId(verbId);
												request.setFrom(from);
												request.setUseLastHarvestDate(useLastHarvestDate);
												request.setUntil(until);
												request.setSetIdentifier(set);
												Utilities.isObjectUnique(storedRequests, request.getRequestURL(), request);
												this.storedRequests.put(request.getRequestURL(), request);
												configureUpdateTriggers(request.getRequestURL(), updateConf);
											} else//TODOException:
												throw new ConfigurationException("this verb action is not supported for harvesting : " + verb);
										}
									} catch (ConfigurationException e) {
										LoggingUtils.logWarn(this.logger, e.getMessage(), e);//logging this so that all configs don't fail
									} catch (SDXException e) {
										LoggingUtils.logWarn(this.logger, e.getMessage(), e);//logging this so that all configs don't fail
									}


								}
							}

							// now we can configure indexation pipeline for these repository
							this.configurePipeline(repoRequestConf);

						}


					}
					if (this.scheduler != null) this.scheduler.start();
				}

			}

		}
	}

	/**Configures time triggers for
	 * stored requests
	 *
	 * @param requestUrl    The request url
	 * @param updateConf    The configuration for updates
	 * @throws ConfigurationException
	 * @see #scheduler
	 * @see #storedRequests
	 */
	protected void configureUpdateTriggers(String requestUrl, Configuration updateConf) throws ConfigurationException {
		if (Utilities.checkString(requestUrl) && updateConf != null) {
			TimeTrigger trigger = new TimeTriggerFactory().createTimeTrigger(updateConf);
			if (trigger != null) {
				if (this.scheduler == null) this.scheduler = new SimpleTimeScheduler();
				this.scheduler.addTrigger(requestUrl, trigger, this);
			}
		}
	}


	/**Configures the repositories
	 * to which data will be stored
	 * based upon their repository url
	 *
	 * @param repoUrl   The repository/data provider url
	 * @param oaiRepoConf The configuration
	 * @throws ConfigurationException
	 */
	protected void configureStoreRepositories(String repoUrl, Configuration oaiRepoConf) throws ConfigurationException {

		//Store repoURL
		if (Utilities.checkString(repoUrl)) {
			String ref = oaiRepoConf.getAttribute(DocumentBaseOAIHarvester.ConfigurationNode.SDX_REPOSITORY, null);
			/*check for the sdxrepository attribute, if it exists, get the repository object and add it to the local hashtable*/
			if (Utilities.checkString(ref)) {
				try {
					Repository repo = null;
					Context appRepos = (Context) getContext().get(ContextKeys.SDX.Application.REPOSITORIES);
					if (appRepos != null)
						repo = (Repository) appRepos.get(ref);
					if (repo == null)
						repo = this.docbase.getRepository(ref);
					if (this.storeRepositoriesRefs == null) this.storeRepositoriesRefs = new Hashtable();
					//populating the hashtable
					Utilities.isObjectUnique(this.storeRepositoriesRefs, repoUrl, repo);
					storeRepositoriesRefs.put(repoUrl, repo);
				} catch (SDXException e) {
					String[] args = new String[1];
					args[0] = ref;
					SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_LOAD_REFERENCED_REPO, args, null);
					throw new ConfigurationException(sdxE.getMessage(), sdxE);
				} catch (ContextException e) {
					throw new ConfigurationException(e.getMessage(), e);
				}
			}
		}

	}

	/**
	 * Check the granularity of an AOI provider : YYYY-MM-DDThh:mm:ssZ or YYYY-MM-DD
	 * @param granularity
	 * @return true or false
	 * @throws ConfigurationException
	 */
	public boolean checkGranularity(String granularity) throws ConfigurationException{
		if(Utilities.checkString(granularity)){
			if(granularity.equals(OAIObject.Node.Value.STRING_GRANULARITY_SECOND)) return true;
			else if(granularity.equals(OAIObject.Node.Value.STRING_GRANULARITY_DAY)) return true;
			else{
				String[] args = new String[2];
				args[0] = granularity;
				args[1] = repoUrl;
				SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_OAI_GRANULARITY, args, null);
				throw new ConfigurationException(sdxE.getMessage(), sdxE);
			}
		} else return false;
	}


	/**Configures the preIndexation pipeline
	 *
	 * @param configuration
	 * @throws ConfigurationException
	 * @see #pipe
	 */
	protected void configurePipeline(Configuration configuration) throws ConfigurationException {
		//at this point, we should have a <sdx:pipeline> element
		Configuration pipeConf = configuration.getChild(Utilities.getElementName(Pipeline.CLASS_NAME_SUFFIX), false);
		//testing if we have something
		if (pipeConf != null) {
			//creating the pipeline and assigning the class field
			this.pipe = new GenericPipeline();
			//setting the super.getLog() for the pipeline object
			this.pipe.enableLogging(this.logger);
			//giving the object the cocoon service manager
			try {
				this.pipe.service(this.manager);
				//pass OAI base properties to the pipe
				this.pipe.contextualize(super.getContext());
				//configuring the pipeline object from 'application.xconf'
				this.pipe.configure(pipeConf);
			} catch (ServiceException e) {
				throw new ConfigurationException(e.getMessage(), e);
			} catch (ContextException e) {
				throw new ConfigurationException(e.getMessage(), e);
			}

		}
	}

	/**Creates a new temporary directory for
	 * writing harvested records before the will
	 * be indexed
	 * @return File
	 * @throws SDXException
	 * @throws IOException
	 */
	protected File getNewTempDir() throws SDXException, IOException {
		File ret = null;
		try {
			ret = (File) getContext().get(Constants.CONTEXT_UPLOAD_DIR);
		} catch (ContextException e) {
			throw new SDXException(logger, SDXExceptionCode.ERROR_GENERIC, null, e);//TODOException: better message here
		}
		String childDir = Utilities.getStringFromContext(ContextKeys.SDX.Application.DIRECTORY_NAME, getContext()) + "_oaiHarvests" + File.separator + this.docbaseId + File.separator + "harvest-" + Utilities.encodeURL(Long.toString(new Date().getTime()), null);
		if (Utilities.checkString(super.resumptionToken))
			childDir += "-" + OAIObject.Node.Name.RESUMPTION_TOKEN + "-" + super.resumptionToken;
		if (ret.canWrite())
			ret = new File(ret, childDir);
		else
			ret = new File(Utilities.getSystemTempDir(), childDir);
		Utilities.checkDirectory(ret.getCanonicalPath(), logger);
		return ret;
	}

	/**Deletes the directory
	 * represented by the tempDir
	 * class field
	 *
	 */
	protected void deleteTempDir() {
		if (tempDir != null) {
			try {
				FileUtil.deleteDirectory(this.tempDir);//deleting the old tempdir if possible
			} catch (IOException e) {
				LoggingUtils.logWarn(logger, e.getMessage(), e);
			}
		}
	}

	/**Establishes the tempDir class field
	 *
	 * @throws SDXException
	 * @throws IOException
	 */
	protected void initTempDir() throws SDXException, IOException {
		this.tempDir = getNewTempDir();
	}

	/**Get's the current date in iso8601 format
	 *
	 * @return String
	 */
	protected String getIsoDate() {
		return fr.gouv.culture.sdx.utils.Date.formatDate(fr.gouv.culture.sdx.utils.Date.getUtcIso8601Date());
	}


	/**Sets up resources to capture an oai record
	 *
	 *
	 * @throws SAXException
	 */
	protected void prepareRecordCapture() throws SAXException {
		try {

			if (this.tempDir == null || !this.tempDir.exists()) initTempDir();
			this.harvestDoc = File.createTempFile("oaiHarvestedRecord", TEMPFILE_SUFFIX, tempDir);
			this.fileOs = new FileOutputStream(this.harvestDoc);

			//Initialise OAI properties for the record
			if (filesProperties == null)
				filesProperties = new Hashtable();
			Properties props = new Properties();
			filesProperties.put(this.harvestDoc.getName(), props);//we can work with file name, automatically generated and unique
			this.currentDatestamp = this.currentMetadtaUrlIdentifier = this.currentOaiIdentifier = this.currentOaiStatus = null;
			this.deleteRecord=false;
			if(cBytes==null){
				cBytes = new XMLSerializer();
				cBytes.enableLogging(logger);
				// transformer configuration
				DefaultConfiguration conf = new DefaultConfiguration("conf");
				DefaultConfiguration cfTf = new DefaultConfiguration("transformer-factory");
				cfTf.setValue(this.transformerFactory);
				((DefaultConfiguration)conf).addChild(cfTf);
				cfTf = new DefaultConfiguration("indent");
				cfTf.setValue(this.transformerIndent);
				((DefaultConfiguration)conf).addChild(cfTf);
				cBytes.configure(conf);
			}
			cBytes.setOutputStream(this.fileOs);
			if(oaiStripper==null){
				// We must strip the OAI envelop
				oaiStripper = new OAIEnvelopStripper();
				oaiStripper.setConsumer(cBytes);
				firstXmlConsumer = oaiStripper;
			}
//			XMLConsumer newConsumer = null;
//			if (firstXmlConsumer != null)
//			newConsumer = new XMLMulticaster(oaiStripper, firstXmlConsumer);
//			else
//			newConsumer = oaiStripper;

//			super.setConsumer(newConsumer);
			super.setConsumer(oaiStripper);

		} catch (IOException e) {
			throw new SAXException(e.getMessage(), e);
		} catch (SDXException e) {
			throw new SAXException(e.getMessage(), e);
		} catch (ConfigurationException e) {
			throw new SAXException(e.getMessage(), e);
		}

	}

	/**Ends the capture of an oai record.
	 *Store properties (dateStamp and OAI identifier) needed for indexation.
	 *
	 *
	 * @throws Exception
	 */
	protected void captureRecord() throws Exception {
		if (this.fileOs != null && this.harvestDoc != null) {

			this.fileOs.flush();
			this.fileOs.close();

			// shoulHarvestCurrentDocument = shouldHarvestDocument(); //we want to know if it's a good idea to harvest this document vs the docs we have in the document base

			if (OAIObject.Node.Value.DELETED.equalsIgnoreCase(currentOaiStatus))
				resetRecordCaptureFields(true);
			else{
				/*Setting OAI properties (dateStamp, OAI identifier) for the current record.
	            Those informations will be used for indexation*/
				Properties oaiProps = (Properties)filesProperties.get(this.harvestDoc.getName());//TODO: do we need to prevent the case where properties are null ? -mp
				if (this.currentDatestamp != null)
					oaiProps.setProperty("datestamp", this.currentDatestamp);
				else oaiProps.setProperty("datestamp", "");
				if (this.currentOaiIdentifier != null)
					oaiProps.setProperty("oaiid", this.currentOaiIdentifier);
				else oaiProps.setProperty("oaiid", "");
				if (this.currentOaiStatus != null)
					oaiProps.setProperty("oaistatus", this.currentOaiStatus);
				else oaiProps.setProperty("oaistatus", "");
				resetRecordCaptureFields(false);
			}

		}
	}

	/**Resets the class fields for record capture
	 * possibility deleting the current <code>harvetDoc</code>
	 * object underlying file
	 *
	 * @param deleteDoc flag for deletion of actual file
	 */
	protected void resetRecordCaptureFields(boolean deleteDoc) {
		if (this.fileOs != null) {
			try {
				this.fileOs.flush();
				this.fileOs.close();
				this.fileOs = null;
			} catch (IOException e) {
				LoggingUtils.logException(logger, e);
			}
		}

		if (this.harvestDoc != null) {
			if (deleteDoc)
				this.harvestDoc.delete();
			this.harvestDoc = null;
		}

		super.setConsumer(super.firstXmlConsumer);//resetting the consumer to the first external consumer
	}


	/**Sets up resources to delete an oai record
	 *Add the record to the list of the records to removed
	 */
	protected void prepareRecordForDeletion() {
		if (this.deletedDocs == null) this.deletedDocs = new ArrayList();
		if (Utilities.checkString(this.currentOaiIdentifier)) {
			XMLDocument deleteDoc = null;
			try {
				deleteDoc = new XMLDocument(this.currentOaiIdentifier);//constructs an XMLDocument. We should pass the sdxdocid, not the sdxoaiid
			} catch (SDXException e) {
				//do nothing here
			}
			if (deleteDoc != null) {
				this.deletedDocs.add(deleteDoc);
				this.noDocsDeleted++;
				try{
					if (this.harvestDoc.exists())
						this.harvestDoc.delete();//delete the file in the temporary directory
				}catch(SecurityException se){
					LoggingUtils.logException(logger, se);
				}
			}
		}
	}


	/**Reads the documents from <code>tempDir</code>
	 * and indexes them in the corresponding document
	 * base, any marked deletions will be carried out
	 * as well
	 *
	 * @return boolean
	 * @throws SDXException
	 * @throws SAXException
	 * @throws ProcessingException
	 * @throws IOException
	 * @see fr.gouv.culture.oai.AbstractOAIHarvester#storeHarvestedData()
	 */
	protected boolean storeHarvestedData() throws ProcessingException, IOException, SDXException, SAXException {
		boolean dataHarvested = false;
		if (docbase != null) {

			//deleting any docs which have been removed from the repo
			if (this.deletedDocs != null && this.deletedDocs.size() > 0 )
				deleteOAIDocuments();

			IndexableDocument[] indexDocs = null;
			//creating our docs from the disk
			if (this.tempDir != null) {
				String[] files = this.tempDir.list();
				ArrayList docs = new ArrayList();
				IndexParameters indexParams = getIndexParameters();
				// at every 1000th document we will index into the base and/or at the end of the group
				int modulusTopLimit = indexParams.getBatchMax() - 1;
				for (int i = 0; i < files.length; i++) {

					String fileName = files[i];

					//Recovers OAI properties (dateStamp, OAI identifier) for the current OAI record
					Properties oaiProps = (Properties)filesProperties.get(fileName);
					this.currentDatestamp = oaiProps.getProperty("datestamp");
					this.currentOaiIdentifier = oaiProps.getProperty("oaiid");
					this.currentOaiStatus = oaiProps.getProperty("oaistatus");

					/*We don't want to indexe document that must be deleted*/
					if (this.currentOaiStatus != null && !OAIObject.Node.Value.DELETED.equalsIgnoreCase(this.currentOaiStatus)) {

						//Constructs OAIDocument wich will be indexed
						OAIDocument metadataDoc = new OAIDocument();
						metadataDoc.setDateString(this.currentDatestamp);
						metadataDoc.setIdentifier(this.currentOaiIdentifier);

						try {
							metadataDoc.setContent(new File(tempDir, fileName).toURL());
							if (docs == null) docs = new ArrayList();
							docs.add(metadataDoc);
							this.noHarvestedDocs++;//keeping track of all additions
						} catch (MalformedURLException e) {
							LoggingUtils.logException(logger, e);
						}

					}

					if ( i == (files.length - 1) || ( i > 0 && (i % modulusTopLimit) == 0 ) ) {
						indexDocs = (IndexableDocument[]) docs.toArray(new IndexableDocument[0]);
						if (indexDocs != null) {
							Repository repo = null;
							if (Utilities.checkString(this.repoUrl) && storeRepositoriesRefs != null)
								repo = (Repository) this.storeRepositoriesRefs.get(this.repoUrl);
							/* FIXME (MP) : A-t-on besoin de consommer les evenements 
                             * SAX envoyes par le processus d'indexation ? 
                             * Si oui, attention ! ce content handler provoque une exception
                             * Caused by: java.io.IOException: Mauvais descripteur de fichier
                             * at java.io.FileOutputStream.writeBytes(Native Method)
                             * [...]
                             * at com.icl.saxon.output.XMLEmitter.writeAttribute(XMLEmitter.java:264)
                             */
                            this.docbase.index(indexDocs, repo, indexParams, /*this*/null);
							dataHarvested = true;
							docs = null;
							indexDocs = null;
						}
					}

				}
			}

		}
		return dataHarvested;
	}


	/**Delete OAI documents from the current document base.
	 *
	 * @throws IOException
	 * @throws ProcessingException
	 * @throws SDXException
	 * @throws SAXException
	 */
	protected void deleteOAIDocuments() throws IOException, ProcessingException, SDXException, SAXException {

		if (this.deletedDocs != null && this.deletedDocs.size() > 0){

			//Construct the SearchLocations for the query
			SearchLocations slocs = new SearchLocations();
			slocs.enableLogging(this.logger);
			//slocs.addIndex(this.docbase.getIndex());
			slocs.addDocumentBase(this.docbase);

			ComplexQuery cq = new ComplexQuery();
			cq.enableLogging(this.logger);
			cq.setUp(slocs, fr.gouv.culture.sdx.search.lucene.query.Query.OPERATOR_OR);

			//boucle sur les documents a supprimer
			for( int i = 0; i < this.deletedDocs.size(); i++ ){
				fr.gouv.culture.sdx.document.XMLDocument doc = (XMLDocument) this.deletedDocs.get(i);
				String sdxoaiid = doc.getId();
				FieldQuery fq = new FieldQuery();
				fq.setUp(slocs, /*sdxoaiid*/sdxoaiid, /*fieldname: sdxoaiid*/OAIDocument.INTERNAL_FIELD_NAME_SDXOAIID);
				cq.addComponent(fq);
			}

			cq.prepare();//prepare the query

			fr.gouv.culture.sdx.search.lucene.query.Results res = cq.execute();//execute the query

			//we have results
			if ( res != null && res.count() > 0) {
				String[] ids = res.getDocIds();
				XMLDocument[] docs = new XMLDocument[ids.length];
				for ( int j = 0; j < docs.length ; j++ ){
					docs[j] = new XMLDocument(ids[j]);
				}
				//delete the documents
				this.docbase.delete(docs, null);
			}

		}

	}


	/**Handles the resumption token by issuing another request
	 * based upon the request from which the resumption token was received.
	 *
	 */
	protected void handleResumptionToken() {
		if (Utilities.checkString(super.resumptionToken) && Utilities.checkString(super.repoUrl)) {
			String verb = "";
			if (this.requestParams!=null)
				verb = this.requestParams.getParameter(OAIObject.Node.Name.VERB, "");
			else if (this.storedRequests.size() > 0){
				verb = ( (OAIRequest)this.storedRequests.get(super.requestUrl) ).getVerbString();
			}
			if (Utilities.checkString(verb)) {
				newRequestUrl = this.repoUrl + OAIRequest.URL_CHARACTER_QUESTION_MARK + OAIObject.Node.Name.VERB + OAIRequest.URL_CHARACTER_EQUALS + verb + OAIRequest.URL_CHARACTER_AMPERSAND + OAIObject.Node.Name.RESUMPTION_TOKEN + OAIRequest.URL_CHARACTER_EQUALS + super.resumptionToken;
				this.resetAllFields();
				try {
					this.initTempDir();//trying to build a tempDir with the  resumptionToken in the dir name
				} catch (SDXException e) {
					LoggingUtils.logException(logger, e);//if this fails we log it as there will be another opportunity during the harvest
				} catch (IOException e) {
					LoggingUtils.logException(logger, e);
				}
				super.resetResumptionToken();//resetting the resumption token here so we don't loop
			}
		}
	}

	//TODO-TEST: this should be preparing for a merge with the metadata, done
	/**Prepares to read a url value from an oai record and
	 * retrieve the XML behind.
	 *@see #identifierName
	 *@see #currentMetadtaUrlIdentifier
	 */
	protected void prepareResourceFromUrlIdentifierCapture() {
		if (Utilities.checkString(this.currentMetadtaUrlIdentifier)) {
			try {
				URL resourceUrl = new URL(this.currentMetadtaUrlIdentifier);
				XMLDocument resource = new XMLDocument();
				resource.setId(this.currentMetadtaUrlIdentifier);
				resource.setContent(resourceUrl);
				this.urlResource = resource;
			} catch (MalformedURLException e) {
				LoggingUtils.logException(logger, e);
			} catch (SDXException e) {
				LoggingUtils.logException(logger, e);
			}

			/*if (docs == null) docs = new ArrayList();
            docs.add(resource);*/


            /*InputStream contents = resource.openStream();
            byte[] chars = new byte[contents.available()];
            contents.read(chars, 0, chars.length);
            contents.close();
            docBytes.
            docBytes.write(chars);*/

		}
	}

	/**Captures the xml from a url taken from an oai record and adds
	 * it to the oai-record as a sibling of the <metadata/> element
	 *
	 */
	protected void captureResourceFromUrlIdentifier() {
		if (this.urlResource != null) {
			try {
				IncludeXMLConsumer include = new IncludeXMLConsumer(super.synchronizedXmlConsumer);
				urlResource.setConsumer(include);
				include.startElement(Framework.SDXNamespaceURI, "urlResource", "sdx:urlResource", new AttributesImpl());
				SAXParser parser = null;
				try {
					parser = (SAXParser) this.manager.lookup(SAXParser.ROLE);
					urlResource.parse(parser);
				} finally {
					if (parser != null)
						this.manager.release(parser);
					this.urlResource = null;
					include.endElement(Framework.SDXNamespaceURI, "urlResource", "sdx:urlResource");
				}
			} catch (SAXException e) {
				LoggingUtils.logException(logger, e);
			} catch (ServiceException e) {
				LoggingUtils.logException(logger, e);
			} catch (SDXException e) {
				LoggingUtils.logException(logger, e);
			}
		}

	}


	/**Resets necessary class fields
	 *
	 */
	protected void resetAllFields() {
		this.deletedDocs = null;
		this.urlResource = null;
		resetRecordCaptureFields(false);
		deleteTempDir();
		this.tempDir = null;
		super.resetAllFields();
	}

	/*Ends the harvest*/
	protected void endHarvest(){
		noHarvestedDocs = noDocsDeleted = 0;
	}


	/**Builds simple index parameters for indexation of
	 * oai records into the undelryi
	 * @return IndexParameters
	 */
	protected IndexParameters getIndexParameters() {
		IndexParameters params = new IndexParameters();
		params.setSendIndexationEvents(IndexParameters.SEND_ALL_EVENTS);
		params.setBatchMax(this.noRecordsPerBatch);
		if ( this.pipe != null ) params.setPipeline(this.pipe);
		else if ( this.docbase != null ) params.setPipeline(this.docbase.getIndexationPipeline());
		params.setPipelineParams(super.getHarvestParameters());
		return params;
	}

	/**Sends the details of stored harvesting requests
	 * to the current consumer
	 *
	 * @throws SAXException
	 */
	public void sendStoredHarvestingRequests() throws SAXException {
		try {
			if (this.storedRequests != null && this.storedRequests.size() > 0) {
				this.acquire();
				super.acquireSynchronizedXMLConsumer();
				super.startElement(Framework.SDXNamespaceURI,
						fr.gouv.culture.sdx.utils.constants.Node.Name.STORED_HARVEST_REQUESTS,
						Framework.SDXNamespacePrefix + ":" + fr.gouv.culture.sdx.utils.constants.Node.Name.STORED_HARVEST_REQUESTS,
						null);
				Enumeration requests = storedRequests.elements();
				if (requests != null) {
					while (requests.hasMoreElements()) {
						OAIRequest request = (OAIRequest) requests.nextElement();
						if (request != null)
							request.toSAX(this);
					}
				}

				super.endElement(Framework.SDXNamespaceURI,
						fr.gouv.culture.sdx.utils.constants.Node.Name.STORED_HARVEST_REQUESTS, Framework.SDXNamespacePrefix + ":" +
						fr.gouv.culture.sdx.utils.constants.Node.Name.STORED_HARVEST_REQUESTS);
			}
		} catch (InterruptedException e) {
			throw new SAXException(e.getMessage(), e);
		} finally {
			super.releaseSynchronizedXMLConsumer();
			this.release();
		}

	}


	/**Triggers a oai request to a repository based
	 * upon a trigger name (also a request url)
	 *
	 * @param triggerName
	 */
	public synchronized void targetTriggered(String triggerName) {
		//verify that we have this trigger defined
		OAIRequest request = (OAIRequest) this.storedRequests.get(triggerName);
		if (request != null) {

			// Test if we have to force the use of the from parameter (in application.xconf) instead of the last haverst date
			boolean useLastHarvestDate = request.getUseLastHarvestDate();
			if ( useLastHarvestDate ){
				try {
					DatabaseEntity dbe = this._database.getEntity(request.getVerbId());
					if ( dbe != null ){
						String lastHarvestDate = dbe.getProperty(LuceneDocumentBaseOAIHarvester.OAI_HARVESTER_LAST_UPDATED);
						request.setFrom(lastHarvestDate);
					}
				} catch (SDXException e) {
					LoggingUtils.logException(logger, e);
				}
			}

			String requestUrl = request.getRequestURL();
			this.receiveSynchronizedRequest(requestUrl, triggerName);// send the request with the original one
			this.endHarvest();
		}

	}


	public void startElement(String s, String s1, String s2, Attributes attributes) throws SAXException {
		if (Utilities.checkString(this.identifierName)) {
			/*looking for an element name or the VALUE of an attribute with the name "name" matching super.indentifierName
			 *so one could create a metadata identifier element like
			 *<super.identifierName>myIdentifierValue</super.identifierName>
			 *or
			 *<anyElementName name="super.identifierName">myIdentifierValue</anyElementName>
			 */
			if (attributes != null && this.identifierName.equals(attributes.getValue(fr.gouv.culture.sdx.utils.constants.Node.Name.NAME)))
				this.captureElemContent = true;
		}
		super.startElement(s, s1, s2, attributes);
	}

	public void endElement(String s, String s1, String s2) throws SAXException {
		if (super.sBuff != null && super.sBuff.length() > 0) {
			String content = super.sBuff.toString();

			if (!OAIObject.Node.Xmlns.OAI_2_0.equals(s)) {
				if (fr.gouv.culture.sdx.utils.constants.Node.Name.FIELD.equals(s1)) {
					super.currentMetadtaUrlIdentifier = content;
					try {
						prepareResourceFromUrlIdentifierCapture();
					} catch (Exception e) {
						//if we can't build the document we don't just fail completely we will continue
						LoggingUtils.logException(logger, e);
					}
				}
			}

		}
		super.endElement(s, s1, s2);
	}

	/**Querys the underlying data structures
	 * based upon current sax flow
	 * position/set class fields and
	 * determines whether an oai record should be
	 * harvested
	 *
	 * @return boolean indicates whether the record should be handled
	 */
	protected boolean shouldHarvestDocument() {
		boolean ret = true;
		try {

			//First control: do we have a previous harvest for this provider/verb ? If so, look for the datestamp of the current oai doc vs. the time of the last harvest
			OAIRequest request = (OAIRequest) this.storedRequests.get(super.requestUrl);
			if (request != null && request.getVerbId()!=null && !request.getVerbId().equals("") && this._database.entityExists(request.getVerbId())){
				DatabaseEntity dbe = this._database.getEntity(request.getVerbId());
				String formattedDate = dbe.getProperty(LuceneDocumentBaseOAIHarvester.OAI_HARVESTER_LAST_UPDATED);
				java.util.Date lastUpdatedDate = fr.gouv.culture.sdx.utils.Date.parseDate(formattedDate);
				java.util.Date currentDocumentDate = fr.gouv.culture.sdx.utils.Date.parseDate(super.currentDatestamp);
				if (currentDocumentDate.getTime() > lastUpdatedDate.getTime())
					return ret;//the timestamp of the current doc is more recent than the last harvest, we should harvest it
			}

			//Second control: search for an sdxdocument which correspond to the current oai id
			else if (Utilities.checkString(super.currentOaiIdentifier)) {

				//The complexQuery
				ComplexQuery cq = new ComplexQuery();
				SearchLocations slocs = new SearchLocations();
				slocs.enableLogging(this.logger);
				slocs.addDocumentBase(this.docbase);
				cq.enableLogging(this.logger);
				cq.setUp(slocs,fr.gouv.culture.sdx.search.lucene.query.Query.OPERATOR_AND);
				//The fieldQuery
				FieldQuery fq = new FieldQuery();
				fq.setUp(slocs, /*sdxoaiid*/super.currentOaiIdentifier, /*fieldname: sdxoaiid*/OAIDocument.INTERNAL_FIELD_NAME_SDXOAIID);
				cq.addComponent(fq);
				//The dateQuery
				DateIntervalQuery dq = new DateIntervalQuery();
				java.util.Date currentOaiDocumentDate= fr.gouv.culture.sdx.utils.Date.parseDate(super.currentDatestamp);
				dq.setUp(slocs, /*sdxoaidate*/OAIDocument.INTERNAL_FIELD_NAME_SDXOAIDATE, /*from*/currentOaiDocumentDate, /*to*/null, true/*bounds inclusive*/);
				cq.addComponent(dq);

				cq.prepare();
				fr.gouv.culture.sdx.search.lucene.query.Results res = cq.execute();

				//we have result(s)
				if (res != null && res.count() == 1 ){//1 doc exists with this sdxoaiid and a sdxoaidate more recent than the current doc, we should not harvest it
					ret = false;
				}
				else if(res != null && res.count() > 1) {//More than 1 result: error
					OAIUtilities.logError(logger, "A problem occured during harvesting: "+res.count()+" documents correspond to the OAI identifier: "+super.currentOaiIdentifier, null);
				}
				else return ret;//no document with this id exists, so we should harvest it
			}
		} catch (SDXException e) {
			LoggingUtils.logException(logger, e);
		}

		return ret;
	}

	/**Saves critical data about a harvest
	 *
	 * @param dataHarvested
	 * @throws SAXException
	 */
	protected void saveCriticalFields(boolean dataHarvested) 
	throws SAXException 
	{
		
		StringBuffer msg = new StringBuffer();
		
		try {

			OAIRequest request = (OAIRequest) this.storedRequests.get(super.requestUrl);
			
			String verbId = request.getVerbId();

			msg.append( "OAI harvest ended. See informations beelow: " );

			if ( request != null && Utilities.checkString(verbId) )
			{

				DatabaseEntity dbe = this._database.getEntity(request.getVerbId());
				if (dbe == null) dbe = new DatabaseEntity(request.getVerbId());
				if(!Utilities.checkString(dbe.getId())) dbe.setId(request.getVerbId());//is it necessary ? -mp

				//Now we have to clear the database
				Property[] props = dbe.getProperties();
				if (props.length > 0){
					for(int i=0 ; i < props.length; i++){
						//Property prop = props[i];
						dbe.deleteProperty(props[i].getName());
					}
				}

				//URL request
				if (Utilities.checkString(super.requestUrl)) {
					dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_REQUEST_URL, super.requestUrl);
					msg.append( "\n\tRequest URL: " + super.requestUrl );
				}

				//URL repo
				if (Utilities.checkString(super.repoUrl)) {
					dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_REPOSITORY_URL, super.repoUrl);
					msg.append( "\n\tRepository URL: " + super.repoUrl );
				}

				//Granularity
				String granularity = request.getGranularity();
				if ( Utilities.checkString(granularity) 
						&& Utilities.checkString(super.responseDate) ) 
				{
					
					String resd = "";
					
					//Store last harvest date with the correct granularity for the current provider
					if(granularity.equals(OAIObject.Node.Value.STRING_GRANULARITY_SECOND)){
						resd = super.responseDate;
					}

					else if(granularity.equals(OAIObject.Node.Value.STRING_GRANULARITY_DAY)){
						resd = super.responseDate.substring(0, super.responseDate.indexOf("T") );
					}
					
					dbe.addProperty( LuceneDocumentBaseOAIHarvester.OAI_HARVESTER_LAST_UPDATED, resd );
					msg.append("\n\tReponse date: " + resd);
					resd = null;

				}

				//Verb request
				String verb = request.getVerbString();
				if (Utilities.checkString(verb)) {

					//Verb as a string
					dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_VERB, verb);
					msg.append("\n\tVerb: " + verb);

					//MetadataPrefix
					String mdPrefix = request.getMetadataPrefix();
					if (Utilities.checkString(mdPrefix)) {
						dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_METADATA_PREFIX, mdPrefix);
						msg.append("\n\tMetadataPrefix: "+mdPrefix);
					}

					//Case : verb=GetRecord
					if (verb.equals(OAIRequest.VERB_STRING_GET_RECORD)) {

						//identifier
						String id = requestParams.getParameter(OAIRequest.URL_PARAM_NAME_IDENTIFIER, null);
						if (Utilities.checkString(id)) {
							dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_IDENTIFIER, id);
							msg.append("\n\tOAI identifier: "+id);
						}
					}

					//Case ListRecords
					if (verb.equals(OAIRequest.VERB_STRING_LIST_RECORDS)) {

						//from
						String from = request.getFrom();
						if (Utilities.checkString(from)) {
							dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_FROM, from);
							msg.append("\n\tFrom: "+from);
						}

						//until
						String until = request.getUntil();
						if (Utilities.checkString(until)) {
							dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_UNTIL, until);
							msg.append("\n\tUntil: "+until);
						}

						//set
						String set = request.getSetIdentifier();
						if (Utilities.checkString(set)) {
							dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_SET, set);
							msg.append("\n\tSet: "+set);
						}

					}

				}

				if (Utilities.checkString(super.resumptionToken)) {
					dbe.addProperty(LuceneDocumentBaseOAIHarvester.OAI_HARVESTER_RESUMPTION_TOKEN, super.resumptionToken);
					msg.append("\n\tResumption token: "+super.resumptionToken);
				}
				
				dbe.addProperty(LuceneDocumentBaseOAIHarvester.NO_DOCS_DELETED, Integer.toString(this.noDocsDeleted));
				dbe.addProperty(LuceneDocumentBaseOAIHarvester.NO_DOCS_HARVESTED, Integer.toString(this.noHarvestedDocs));
				
				msg.append("\n\tDocuments deleted: "+this.noDocsDeleted);
				msg.append("\n\tDocuments harvested: "+this.noHarvestedDocs);

				this._database.update(dbe);
				this._database.optimize();
				
				OAIUtilities.logInfo(logger, msg.toString());

			}

		} catch (Exception e) {
			throw new SAXException(e.getMessage(), e);
		} finally {
			msg = null;
		}
	}

	/**Generates an id to associate
	 * with a harvest
	 *
	 * @return String
	 */
	protected String generateNewHarvestId() {
		return this.harvesterIdGen.generate();
	}

	/**Sends sax events to the current consumer
	 * with summary details of the all the past harvests
	 *
	 * @throws SAXException
	 */
	public void sendPastHarvestsSummary() throws SAXException {
		//build a query get the doc
		try {
			DatabaseEntity[] dbes = this._database.getEntities();
			if (dbes != null && dbes.length > 0) {
				this.acquire();
				acquireSynchronizedXMLConsumer();
				super.startElement(Framework.SDXNamespaceURI, fr.gouv.culture.sdx.utils.constants.Node.Name.PREVIOUS_HARVESTS,
						Framework.SDXNamespacePrefix + ":" + fr.gouv.culture.sdx.utils.constants.Node.Name.PREVIOUS_HARVESTS, null);
				for (int i = 0; i < dbes.length; i++) {
					DatabaseEntity dbe = dbes[i];
					AttributesImpl atts = new AttributesImpl();
					String repoUrl = null;
					if (dbe != null) {
						Property[] props = dbe.getProperties();
						if (props != null) {
							for (int j = 0; j < props.length; j++) {
								Property prop = props[j];
								if (prop != null) {
									String propName = prop.getName();
									if (Utilities.checkString(propName) && !LuceneDocumentBaseOAIHarvester.OAI_HARVEST_ID.equals(propName)) {
										String propVal = prop.getValue();
										if (propName.equals(LuceneDocumentBaseOAIHarvester.OAI_REPOSITORY_URL)) {
											repoUrl = propVal;
										} else {
											if (Utilities.checkString(propVal))
												atts.addAttribute("", propName, propName, OAIObject.Node.Type.CDATA, propVal);
										}
									}
								}

							}
						}

					}
					sendElement(OAIObject.Node.Xmlns.OAI_2_0, OAIObject.Node.Name.REQUEST, OAIObject.Node.Name.REQUEST,
							atts, repoUrl);
				}
				super.endElement(Framework.SDXNamespaceURI, fr.gouv.culture.sdx.utils.constants.Node.Name.PREVIOUS_HARVESTS,
						Framework.SDXNamespacePrefix + ":" + fr.gouv.culture.sdx.utils.constants.Node.Name.PREVIOUS_HARVESTS);
			}

		} catch (/*IO*/Exception e) {
			throw  new SAXException(e.getMessage(), e);
		}   /*   Utilities.logException(super.getLog(), e);TODO: what is the best way to handle this?
        } catch (SDXException e) {
            try {
                e.toSAX(this);
            } catch (ProcessingException e1) {
                Utilities.logException(super.getLog(), e1);
            }
        } */ finally {
        	releaseSynchronizedXMLConsumer();
        	this.release();
        }


	}

	/**Retrieves the time when the harvester was last updated
	 *
	 * @return Date
	 */
	public Date lastUpdated() {
		return this.docbase.lastModificationDate();
	}

	/**Destroys all summary data pertaining to past harvests
	 * but not the actual oai records harvested
	 *
	 */
	public void purgePastHarvestsData() {
		try {
			DatabaseEntity[] dbes = this._database.getEntities();
			for (int i = 0; i < dbes.length; i++) {
				DatabaseEntity dbe = dbes[i];
				this._database.delete(dbe);
			}
			this._database.optimize();
		} catch (SDXException e) {
			LoggingUtils.logException(logger, e);
		}
	}

	/**Stores data about harvesting failures caused
	 * by problems other than  oai errors sent from
	 * a queried repository
	 *
	 * @param e Exception
	 * @see fr.gouv.culture.oai.AbstractOAIHarvester#storeFailedHarvestData(java.lang.Exception)
	 */
	protected void storeFailedHarvestData(Exception e) {
		try {
			//TODO: call this method when harvesting fails/major exception thrown
			//create a special property "failedHarvest"
			DatabaseEntity dbe = new DatabaseEntity(generateNewHarvestId());
			String message = "noMessage";
			if (e != null) message = e.getMessage();
			dbe.addProperty(OAI_FAILED_HARVEST, message);
			//create a _database entity and store all relevant class fields
			dbe.addProperty(OAI_REQUEST_URL, this.requestUrl);
			//save it to the database
			this._database.update(dbe);
			this._database.optimize();
			//TODO:later perhaps we can try to auto-reexecute these failed request, keying on the OAI_FAILED_HARVEST property ???
		} catch (SDXException e1) {
			LoggingUtils.logException(logger, e1);
		}
	}


	/** Save the timeStamp of the Harvester
	 * @see fr.gouv.culture.sdx.utils.save.Saveable#backup(fr.gouv.culture.sdx.utils.save.SaveParameters)
	 */
	public void backup(SaveParameters save_config) throws SDXException {
		if(save_config != null)
			if(save_config.getAttributeAsBoolean(Saveable.ALL_SAVE_ATTRIB,false))
			{
				String oai_path = DocumentBase.ConfigurationNode.OAI_HARVESTER;
				File oai_dir = new File(save_config.getStoreCompletePath()+ File.separator +oai_path);
				if(!oai_dir.exists())
					oai_dir.mkdir();
			}
	}
	/** Restore the timeStamp of the Harvester
	 * @see fr.gouv.culture.sdx.utils.save.Saveable#restore(fr.gouv.culture.sdx.utils.save.SaveParameters)
	 */
	public void restore(SaveParameters save_config) throws SDXException {
		// TODO Auto-generated method stub
	}

	/** Close OAI harvester.
	 */
	public void close(){
		if(scheduler!=null && storedRequests!=null){
			String tn=null;
			for(Enumeration e = storedRequests.keys() ; e.hasMoreElements() ;){
				tn = (String) e.nextElement();
				if (Utilities.checkString(tn)){
					scheduler.removeTrigger(tn);
				}
			}
			storedRequests = new Hashtable();
			scheduler = null;
		}
	}

}
