/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.oai;

import fr.gouv.culture.sdx.exception.SDXException;
import org.apache.avalon.framework.configuration.Configurable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;


/**<p>This class is a SAX base repository build open Cocoon's
 * XMLPipe interface. SAX events are generated when methods
 * corresponding to OAI verbs are called and sent to the
 * appropriate consumer.</p>
 *
 * <p>NOTE: the element corresponding to the verb name should be provided by
 * these methods as in the sax stream if parameters are verified before
 * response generation error elements will appear in the oai response
 * and not within the verb response.</p>
 *
 * <p>The SAX output is currently OAI-PMH 2.0 conformant.</p>
 *
 */
public interface OAIRepository extends OAIObject, Configurable {


    /*
    4.1. GetRecord
    Summary and Usage Notes
    This verb is used to retrieve an individual metadata record from a repository.  Required arguments specify the identifier of the item from which the record is requested and the format of the metadata that should be included in the record. Depending on the level at which a repository tracks deletions, a header with a "deleted" value for the status attribute may be returned, in case the metadata format specified by the metadataPrefix is no longer available from the repository or from the specified item.

    Arguments
    identifier a required argument that specifies the unique identifier  of the item in the repository from which the record must be disseminated.
    metadataPrefix a required argument that specifies the metadataPrefix of the format that should be included in the metadata part of the returned record .  A record should only be returned if the format specified by the metadataPrefix can be disseminated from the item identified by the value of the identifier argument. The metadata formats supported by a repository and for a particular record can be retrieved using the ListMetadataFormats request.
    Error and Exception Conditions
    badArgument - The request includes illegal arguments or is missing required arguments.
    cannotDisseminateFormat - The value of the metadataPrefix argument is not supported by the item identified by the value of the identifier argument
    idDoesNotExist - The value of the identifier argument is unknown or illegal in this repository.
    */
    /**Send's xml data via SAX events to the Consumer, etc. see Cocoon's XMLProducer interface
     *
     *
     * @param request
     * @throws SAXException
     */
    void getRecord(OAIRequest request) throws SAXException;


    /*
    4.2. Identify
    Summary and Usage Notes
    This verb is used to retrieve information about a repository.  Some of the information returned is required as part of the OAI-PMH.  Repositories may also employ the Identify verb to return additional descriptive information.

    Arguments
    None

    Error and Exception Conditions
    badArgument - The request includes illegal arguments.
    Response Format
    The response must include one instance of the following elements:

    repositoryName : a human readable name for the repository;
    baseURL : the base URL of the repository;
    protocolVersion : the version of the OAI-PMH supported by the repository;
    earliestDatestamp : a UTCdatetime that is the guaranteed lower limit of all datestamps recording changes, modifications, or deletions in the repository. A repository must not use datestamps lower than the one specified by the content of the earliestDatestamp element. earliestDatestamp must be expressed at the finest granularity supported by the repository.
    deletedRecord : the manner in which the repository supports the notion of deleted records . Legitimate values are no ; transient ; persistent with meanings defined in the section on deletion .
    granularity: the finest harvesting granularity supported by the repository. The legitimate values are YYYY-MM-DD and YYYY-MM-DDThh:mm:ssZ with meanings as defined in ISO8601.
    The response must include one or more instances of the following element:

    adminEmail : the e-mail address of an administrator of the repository.
    The response may include multiple instances of the following optional elements:

    compression : a compression encoding supported by the repository. The recommended values are those defined for the Content-Encoding header in Section 14.11 of RFC 2616 describing HTTP 1.1.  A compression element should not be included for the identity encoding, which is implied.
    description : an extensible mechanism for communities to describe their repositories.  For example, the description container could be used to include collection-level metadata in the response to the Identify request. Implementation Guidelines are available to give directions with this respect. Each description container must be accompanied by the URL of an XML schema describing the structure of the description container.
    */
    /**Send's xml data via SAX events to the Consumer, etc. see Cocoon's XMLProducer interface
     *
     *
     * @param request
     * @throws SAXException
     */
    void identify(OAIRequest request) throws SAXException;

    /*One of the below is required by the OAI protocol within a response from a repository queried with the "Identify" verb*/
    String getGranularity();

    String getRepositoryName();

    String getEarliestDatestamp();

    String getDeletedRecord();

    String getProtocolVersion();

    String getBaseURL();
    
    String getResumptionToken();


    /**Required element within a response from a repository queried with the "Identify" verb
     *
     */
    String[] getAdminEmails();

    /*The below are optional and repeatable within a response from a repository queried with the "Identify" verb */
    String getCompression();


    /**Sends the description xml for the repository to the provided content handler
     *
     * @param handler   The handler to feed with events
     */
    void getDescription(ContentHandler handler) throws SAXException;

    /*
    4.3. ListIdentifiers
    Summary and Usage Notes
    This verb is an abbreviated form of ListRecords, retrieving only headers rather than records. Optional arguments permit selective harvesting of headers based on set membership and/or _datestamp. Depending on the repository's support for deletions, a returned header may have a status attribute of "deleted" if a record matching the arguments specified in the request has been deleted.

    Arguments
    from an optional argument with a UTCdatetime value , which specifies a lower bound for _datestamp-based selective harvesting.
    until an optional argument with a UTCdatetime value , which specifies a upper bound for _datestamp-based selective harvesting.
    metadataPrefix a required argument, which specifies that headers should be returned only if the metadata format matching the supplied metadataPrefix is available or, depending on the repository's support for deletions, has been deleted. The metadata formats supported by a repository and for a particular item can be retrieved using the ListMetadataFormats request.
    set an optional argument with a setSpec value , which specifies set criteria for selective harvesting.
    resumptionToken an exclusive argument with a value that is the flow control token returned by a previous ListIdentifiers request that issued an incomplete list.
    Error and Exception Conditions
    badArgument - The request includes illegal arguments or is missing required arguments.
    badResumptionToken - The value of the resumptionToken argument is invalid or expired.
    cannotDisseminateFormat - The value of the metadataPrefix argument is not supported by the repository.
    noRecordsMatch- The combination of the values of the from, until , and set arguments results in an empty list.
    noSetHierarchy - The repository does not support sets.
    */
    /**Send's xml data via SAX events to the Consumer, etc. see Cocoon's XMLProducer interface
     *
     *
     * @param request
     * @throws SAXException
     */
    void listIdentifiers(OAIRequest request) throws SAXException;


    /*
   4.4. ListMetadataFormats
   Summary and Usage Notes
   This verb is used to retrieve the metadata formats available from a repository.  An optional argument restricts the request to the formats available for a specific item.

   Arguments
   identifier an optional argument that specifies the unique identifier of the item for which available metadata formats are being requested.  If this argument is omitted, then the response includes all metadata formats supported by this repository.  Note that the fact that a metadata format is supported by a repository does not mean that it can be disseminated from all items in the repository.
   Error and Exception Conditions
   badArgument - The request includes illegal arguments or is missing required arguments.
   idDoesNotExist - The value of the identifier argument is unknown or illegal in this repository.
   noMetadataFormats - There are no metadata formats available for the specified item.
   */
    /**Send's xml data via SAX events to the Consumer, etc. see Cocoon's XMLProducer interface
     *
     *
     * @param request
     * @throws SAXException
     */
    void listMetadataFormats(OAIRequest request) throws SAXException;


    /*
    4.5. ListRecords
    Summary and Usage Notes
    This verb is used to harvest records from a repository.  Optional arguments permit selective harvesting of records based on set membership and/or _datestamp. Depending on the repository's support for deletions, a returned header may have a status attribute of "deleted" if a record matching the arguments specified in the request has been deleted. No metadata will be present for records with deleted status.

    Arguments
    from an optional argument with a UTCdatetime value , which specifies a lower bound for _datestamp-based selective harvesting.
    until an optional argument with a UTCdatetime value , which specifies a upper bound for _datestamp-based selective harvesting.
    set an optional argument with a setSpec value , which specifies set criteria for selective harvesting.
    resumptionToken an exclusive argument with a value that is the flow control token returned by a previous ListRecords request that issued an incomplete list.
    metadataPrefix a required argument (unless the exclusive argument resumptionToken is used) that specifies the metadataPrefix of the format that should be included in the metadata part of the returned records.   Records should be included only for items from which the metadata format
    matching the metadataPrefix can be disseminated. The metadata formats supported by a repository and for a particular item can be retrieved using the ListMetadataFormats request.
    Error and Exception Conditions
    badArgument - The request includes illegal arguments or is missing required arguments.
    badResumptionToken - The value of the resumptionToken argument is invalid or expired.
    cannotDisseminateFormat - The value of the metadataPrefix argument is not supported by the repository.
    noRecordsMatch - The combination of the values of the from, until, set, and metadataPrefix  arguments results in an empty list.
    noSetHierarchy - The repository does not support sets.
    */
    /**Send's xml data via SAX events to the Consumer, etc. see Cocoon's XMLProducer interface
     *
     *
     * @param request
     * @throws SAXException
     */
    void listRecords(OAIRequest request) throws SAXException;


    /*
    4.6. ListSets
    Summary and Usage Notes
    This verb is used to retrieve the set structure of a repository, useful for selective harvesting.

    Arguments
    resumptionToken an exclusive argument with a value that is the flow control token returned by a previous ListSets request that issued an incomplete list
    Error and Exception Conditions
    badArgument - The request includes illegal arguments or is missing required arguments.
    badResumptionToken - The value of the resumptionToken argument is invalid or expired.
    noSetHierarchy - The repository does not support sets.
    */
    /**Send's xml data via SAX events to the Consumer, etc. see Cocoon's XMLProducer interface
     *
     *
     * @param request
     * @throws SAXException
     */
    void listSets(OAIRequest request) throws SAXException;


    /**Verifies the request parameters respective of the
     * verb provided in the request and the repository's
     * level of support for certain optional features
     * like "resumptionToken"s and "set"s
     *
     * note this method should return SAX events to any
     * XMLConsumer set if errors exist within the request parameters
     *
     * @param request The request objec
     */
    boolean verifyParameters(OAIRequest request) throws SAXException;

    /**Removes a oai deleted record entry FROM the repository
     *
     *@param id The id of the record to remove
     */
    void removeDeletedRecord(String id) throws SDXException;

    /**Adds a oai deleted record entry TO the repository
     *
     *@param id The id of the record to remove
     */
    void addDeletedRecord(String id) throws SDXException;

    /**Deletes all deleted records FROM the repository
     *
     */
    void purgeDeletedRecords();


}
