
import java.awt.*;
import java.awt.event.*;

class JGWAdapter extends WindowAdapter {
  private Component obj;

  JGWAdapter(Component obj) {
    super();
    this.obj = obj;
  }

  public void windowClosing(WindowEvent event) {
    System.out.println("jgraph dispose");
    obj.setVisible(false);
  }
}


class JavaJGraph extends Frame
  implements ActionListener,
  AdjustmentListener
{
  int color;
  int width, height;		// window size 

  int xs, ys;			// scale factors

  JavaJMemCanvas canvas;
  Scrollbar scrollbar;

  Font axis_font;

  private static final int PX_PER_POINT = 4;
  private static final int ARR_CHUNK = 1020;
  private static final int AXIS_SIZE = 60;

  double x[];
  double y[];

  double maxx, minx;
  double maxy, miny;

  int arr_cnt, arr_max;
  int arr_cur;
  int arr_visible;		// array range represented on screen

  String x_axis_label, y_axis_label;
  Image y_axis_image;
  String title;

  JavaJGraph(int w, int h) {
    super();

    System.out.println("JavaJGraph init");
    width = w; height = h;
    color = 0;

    arr_cnt = 0;
    arr_cur = -1;		// -1 => follow the latest points
    arr_max = ARR_CHUNK;
    x = new double[ARR_CHUNK];
    y = new double[ARR_CHUNK];

    maxx = maxy = minx = miny = 0.0;

    x_axis_label = "X Axis";
    y_axis_label = "Y Axis";
    y_axis_image = null;
    title = "";

    axis_font = new Font("TimesRoman", Font.BOLD, 10);

    setSize(w, h);

    // give us a canvas to stretch over the whole thing
    canvas = new JavaJMemCanvas();
    add("Center", canvas);

    scrollbar = new Scrollbar(Scrollbar.HORIZONTAL);
    scrollbar.addAdjustmentListener(this);
    add("South", scrollbar);

    setBackground(Color.white);
    setVisible(true);

    if (false)
      addWindowListener(new JGWAdapter(this));
    System.out.println("JavaJGraph leaving init");
  }

  // we store the data we get in a vector of doubles

  // make sure our data vector is of at least length "n"
  private synchronized void setLength(int n) {
    if (n > arr_max) {
      int new_size = arr_max;

      System.out.println("increasing array");
      while (new_size <= n)
	new_size += new_size;

      double new_x[] = new double[new_size];
      double new_y[] = new double[new_size];
      for (int i=0; i<arr_cnt; i++) {
	new_x[i] = x[i];
	new_y[i] = y[i];
      }

      x = new_x;
      y = new_y;
    }

    arr_cnt = n;
  }

  // manage our data values
  private synchronized void addData(double xval, double yval) {
    int idx = arr_cnt;

    setLength(idx+1);
    x[idx] = xval;
    y[idx] = yval;

    if (idx == 0 || xval > maxx)
      maxx = xval;
    if (idx == 0 || yval > maxy)
      maxy = yval;
    if (idx == 0 || xval < minx)
      minx = xval;
    if (idx == 0 || yval < miny)
      miny = yval;

  }

  public synchronized void update() {
    // grab the canvas; get its dimensions
    Graphics ig = canvas.imageGraphics();
    Dimension size = canvas.getSize();

    System.out.println("drawing on "+ig+" size="+size.width+"x"+size.height);

    // clear the canvas
    ig.clearRect(0, 0, size.width, size.height);

    int arr_start;	// first index
    int arr_end;       	// last index

    // make a plot area, leaving AXIS_SIZE space for the axis info
    int xorg = AXIS_SIZE;
    int yorg = 0;
    int plot_width = size.width - AXIS_SIZE;
    int plot_height = size.height - AXIS_SIZE;

    if (plot_width < 0 || plot_height < 0)
      return;

    // scale the data and draw as much as fits
    if (arr_cur == -1) {
      // count backwards and fit as much as you can on the screen
      arr_end = arr_cnt;
      int last_x = plot_width;
      double end_pt = x[arr_end-1];

      for (arr_start = arr_end-1; arr_start > 0; arr_start--) {
	if (last_x - ((end_pt-x[arr_start]) * PX_PER_POINT) < 0)
	  break;
      }

      arr_visible = arr_end - arr_start;
      scrollbar.setValues(arr_start, arr_end-arr_start, 0, arr_cnt);
    } else {
      // count forwards from arr_cur
      arr_start = arr_cur;
      int last_x = 0;
      for (arr_end = arr_cur+1; arr_end < arr_cnt; arr_end++) {
	last_x += (x[arr_end]-x[arr_end-1]) * PX_PER_POINT;
	if (last_x >= plot_width) break;
      }

      arr_visible = arr_end - arr_cur;
      scrollbar.setValues(arr_cur, arr_end-arr_cur, 0, arr_cnt);
    }

    Graphics plotG = ig.create(xorg, yorg, plot_width, plot_height);

    // figure out how many of these points will fit on the screen
    int x_cnt = arr_end-arr_start;
    int p_x[] = new int[x_cnt];
    int p_y[] = new int[x_cnt];

    // assume our range starts at zero; leave a little headroom
    double y_range = (maxy * 4.0) / 3.0;
    double x_range = x[arr_end-1] - x[arr_start];

    // save an x,y pair translated to screen coordinates
    for (int i=arr_start; i < arr_end; i++) {
      p_x[i-arr_start] = (int)((x[i]-x[arr_start]) * PX_PER_POINT);
      p_y[i-arr_start] = plot_height - ((int)(plot_height * (y[i]/y_range)));
    }
      
    // draw the points
    plotG.drawPolyline(p_x, p_y, x_cnt);
    plotG.dispose();

    // draw south and west axes
    FontMetrics fm = ig.getFontMetrics(axis_font);
    int x_len = p_x[x_cnt-1];

    // draw the axis and some hash lines
    draw_y_axis(ig, xorg, yorg, yorg+plot_height, y_range, fm);
    draw_x_axis(ig, yorg+plot_height, xorg, xorg+x_len, x[arr_start], 
		x_range, fm);

    if (y_axis_label != null) {
      if (y_axis_image == null) {

	// rotate the text 90 degrees; we have to generate an image
	// that has the rotated text
	JavaJRotText jrt = new JavaJRotText(y_axis_label, 
					    JavaJRotText.R_90,
					    axis_font);
	y_axis_image = jrt.getImage(this, ig, size.height, AXIS_SIZE);
      }

      if (y_axis_image != null) {
	// draw the y axis label image (see above) in the middle of	
	// the y axis area
	int y = (size.height - y_axis_image.getHeight(null)) / 2;
	if (y < 0) y = 0;
	ig.drawImage(y_axis_image, 4, y, null);
      }
    }
    
    if (x_axis_label != null) {

      // draw the x axis label in the middle of a spot just up from
      // the bottom of the screen
      int x_label_baseline = fm.getMaxDescent();
      int x_label_width = fm.stringWidth(x_axis_label);
      int x = (size.width - x_label_width)/2;
      if (x < 0) x = 0;

      ig.setFont(axis_font);
      ig.drawString(x_axis_label, x, size.height-x_label_baseline );
    }

    repaint();
  }

  private void draw_x_axis(Graphics g, int y, 
			   int x_start, int x_end, // pixel start, end
			   double data_val, 	   // data start, end
			   double range, 
			   FontMetrics fm) {
    //  x  a x i s
    g.drawLine(x_start, y, x_end-1, y);

    // figure out the order of magnitude for the axis
    double exp = Math.log(range) / Math.log(10);
    exp = Math.floor(exp);
    double inc = Math.pow(10.0, exp);

    // draw some hashes and coordinate labels

    // calculate the space between hashes; 10 little hashes per big one
    double hash_delta = inc * ((double)(x_end-x_start) / range);
    
    int ascent = fm.getMaxAscent();
    data_val += inc;
    if (hash_delta > 0.0) {
      for (double x = x_start+hash_delta, cnt = 1; 
	   x < x_end;
	   x += hash_delta, cnt++, data_val += inc) {

	// major lines
	g.drawLine((int)x, y,  (int)x, y+5);

	// and their labels
	String label = Double.toString(data_val);
	// truncate to a couple of interesting digits
	int digwid = (int)(Math.abs(exp))+3;
	String tlab = (label.length() > digwid) ? 
	  label.substring(0, digwid) :
	  label;


	int labwid = fm.stringWidth(tlab);
	g.drawString(tlab, (int)x-(labwid/2), y+ascent+8); 
      }
    }
  }

  private void draw_y_axis(Graphics g, int x, 
			   int y_start, int y_end, 
			   double range, FontMetrics fm) {

    //   y  a x i s

    g.drawLine(x, y_start, x, y_end-1);

    // figure out the order of magnitude for the axis
    double exp = Math.log(range) / Math.log(10);
    exp = Math.floor(exp);
    double inc = Math.pow(10.0, exp);

    // draw some hashes and coordinate labels

    // calculate the space between hashes; 10 little hashes per big one
    double hash_delta = inc * ((double)(y_end-y_start) / range);
    
    double data_val = 0.0 + inc;
    if (hash_delta > 0.0) {
      for (double y = y_end-hash_delta, cnt = 1; 
	   y >= 0; 
	   y -= hash_delta, cnt++, data_val += inc) {

	// major lines
	g.drawLine(x, (int)y,  x-5, (int)y);

	// and their labels
	String label = Double.toString(data_val);
	if (fm.getMaxAscent() < y) {
	  // if we have enough headroom, print the label

	  // truncate to a couple of interesting digits
	  int digwid = (int)(Math.abs(exp))+3;
	  String tlab = (label.length() > digwid) ? 
	    label.substring(0, digwid) :
	    label;

	  int labwid = fm.stringWidth(tlab);
	  g.drawString(tlab, x-8-labwid, (int)y); 
	}
      }
    }
  }

  public void setRanges(double minx, double maxx, double miny, double maxy) {
    this.minx = minx;
    this.maxx = maxx;
    this.miny = miny;
    this.maxy = maxy;
  }

  public void setScaleMode(int xs, int ys) {
    System.out.println("JGraph: Scale mode "+xs+","+ys);
  }

  public void setTitle(String title) {
    this.title = title;
  }

  public void setAxisLabels(String xl, String yl) {
    this.x_axis_label = xl;
    this.y_axis_label = yl;
  }

  // set the length of this vector
  public synchronized void setElemLen(String elemName, int n) {
    int old_len = arr_cnt;

    setLength(n);

    // pad out extensions with zeros
    for (int i=old_len; i<n; i++) {
      x[i] = 0.0;
      y[i] = 0.0;
    }
  }

  // add another data point to this graph
  public void addElem(String elemName, double x, double y) {
    addData(x, y);
  }

  // remove the n'th data point from this graph
  public void dropElem(String elemName) {
    System.out.println("JGraph: drop "+elemName);
  }


  public void paint(Graphics g) {
    canvas.repaint();
    g.dispose();
  }



  public synchronized void adjustmentValueChanged(AdjustmentEvent event) {

    int val = event.getValue();
    int visible = arr_visible;
    int cnt = arr_cnt;

    // if we go to the end of the range, go back to tracking
    // the graph progress
    if (val >= cnt-visible)
      arr_cur = -1;
    else
      arr_cur = val;
  }

  public void actionPerformed(ActionEvent event) {
    Object source = event.getSource();
    System.out.println("event!" + event);
  }


}
