/* Copyright (c) 2011 Daniel Thiele, http://www.ib-dt.de
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id$ */
/**
 * @file
 * @brief ....
 * @_addtogroup grpApp...
 */


/* === includes ============================================================ */
#include <stdint.h>
#include <stdio.h>
#include <util/delay.h>
#include <util/atomic.h>

#include <board.h>
#include <ioutil.h>
#include <timer.h>
#include <hif.h>
#include <transceiver.h>
#include <radio.h>

#include "reg_hmc5883l.h"
#include "reg_lis331hh.h"
#include "reg_bmp085.h"

#include "sensors.h"
#include "i2c.h"

/* === macros ============================================================== */

#define PINSTATE(port, bp) ((port&_BV(bp))>>bp)

/* === types =============================================================== */

/* === globals ============================================================= */

/* Host addr for I2C debug output */
static const uint8_t i2c_hostaddr=0xB8;

/* === prototypes ========================================================== */

/* === functions =========================================================== */

static int i2c_putc(int c, FILE *stream) {
	uint8_t stat;
	stat = i2c_startcondition();
	if (TW_START == stat) {
		stat = i2c_writebyte(i2c_hostaddr | TW_WRITE);
		if (TW_MT_SLA_ACK == stat) {
			i2c_writebyte(c & 0x00FF);
		}
	}
	stat = i2c_stopcondition();
	return c;
}


static void selftest_trxpin_rstn(void)
{
	uint8_t testval = 0xB5;
	const uint8_t testaddr = RG_IEEE_ADDR_0;
	const uint8_t rstval = 0x00;

	TRX_RESET_HIGH();
	trx_reg_write(testaddr, testval);

	TRX_RESET_LOW();
	_delay_us(TRX_RESET_TIME_US);
	TRX_RESET_HIGH();

	printf("TRX.PIN.RSTN..");
	if((testval=trx_reg_read(testaddr)) != rstval){
		printf("FAIL\n");
		printf("Read %02X != %02X", testval, rstval);
	}else{
		printf("PASS\n");
	}
}

static void selftest_trxpin_slptr(void)
{
	uint16_t timeout = 0xFFFF;

	trx_reg_write(RG_TRX_STATE, CMD_FORCE_TRX_OFF);
	trx_reg_write(RG_IRQ_MASK, TRX_IRQ_TRX_END);

	trx_reg_read(RG_IRQ_STATUS);	/* flush */

	DI_TRX_IRQ();
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE){
		trx_bit_write(SR_TX_PWR, 0x0F);	/* lowest power */
		trx_reg_write(RG_TRX_STATE, CMD_PLL_ON);

		/* wait for PLL_LOCK, no checking here */
		_delay_ms(10);

		TRX_SLPTR_HIGH();
		_delay_us(10);
		TRX_SLPTR_LOW();

		while( (!(trx_reg_read(RG_IRQ_STATUS) & TRX_IRQ_TRX_END)) && --timeout);
	}
	EI_TRX_IRQ();
	trx_reg_write(RG_TRX_STATE, CMD_FORCE_TRX_OFF);

	printf("TRX.PIN.SLPTR..");
	if(!timeout){
		printf("FAIL\n");
	}else{
		printf("PASS\n");
	}
}

void selftest_trxpin_irq(void)
{
	uint8_t timeout = 20;	/* 50us delay, results in 1ms max. time */

	trx_reg_write(RG_TRX_STATE, CMD_FORCE_TRX_OFF);
	trx_reg_write(RG_IRQ_MASK, TRX_IRQ_PLL_LOCK);

	trx_reg_read(RG_IRQ_STATUS);	/* flush */

	ATOMIC_BLOCK(ATOMIC_RESTORESTATE){
		trx_reg_write(RG_TRX_STATE, CMD_PLL_ON);
		_delay_us(50);
		while( (!(trx_reg_read(RG_IRQ_STATUS) & TRX_IRQ_PLL_LOCK)) && --timeout);
	}
	trx_reg_write(RG_TRX_STATE, CMD_FORCE_TRX_OFF);

	printf("TRX.PIN.IRQ..");
	if(!timeout){
		printf("FAIL\n");
	}else{
		printf("PASS\n");
	}
}

void selftest_lis331(void)
{
	uint8_t status;
	uint8_t wr;

	wr = 0x2F;
	i2c_write(SENSORS_I2CADDR_LIS3331, RG_LIS331HH_CTRL_REG1, &wr, 1);

	printf("-- LIS331HH --");
	for(uint8_t i=0;i<20;i++){
		i2c_read(SENSORS_I2CADDR_LIS3331, RG_LIS331HH_STATUS_REG, &status, 1);
	}
}

void selftest(void)
{
	static FILE i2cstream = FDEV_SETUP_STREAM(i2c_putc, NULL, _FDEV_SETUP_WRITE);

	i2c_init();

	stdin = stdout = stderr = &i2cstream;

	printf("\n\n-Rose231 Selftest-\n");
	printf("CVS $Id$\n");
	printf("BUILD %s %s\n", __DATE__, __TIME__);
	printf("--Probing I2C [0x00 .. 0x7F]--\n");

	for(uint8_t i=0;i<0x7F;i++)
	{
		i2c_startcondition();
		uint8_t stat = i2c_writebyte(i<<1 | TW_WRITE);
		i2c_stopcondition();

		_delay_us(100); /* WTF */

		if(TW_MT_SLA_ACK == stat) {
			printf("I2C.Device at %02X\n", i<<1);
		}
	}

	printf("--Probing TRX--\n");

	/* Init MCU peripherals */
	trx_io_init( SPI_RATE_1_2);
	TRX_IRQ_INIT();
	DI_TRX_IRQ();

	TRX_RESET_LOW();
	TRX_SLPTR_LOW();
	DELAY_US( TRX_RESET_TIME_US);
	TRX_RESET_HIGH();

	selftest_trxpin_rstn();
	selftest_trxpin_slptr();
	selftest_trxpin_irq();

	printf("TRX.PART_NUM:%02X, TRX.VER_NUM:%02X\n", trx_reg_read(RG_PART_NUM), trx_reg_read(RG_VERSION_NUM));

	selftest_lis331();
}

/* EOF */
