/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* The function keys_debounced() use the code given in Peter Fleury's
   avr-gcc examples. File: avrgcc-examples/debounce_keys.c */

/****************************************************************************
 Title:    Debouncing 8 Keys
 Author:   Peter Fleury <pfleury@gmx.ch> http://jump.to/fleury,
           based on algorithm of Peter Dannegger <danni@specs.de>
 Date:     December 2003
 Software: AVR-GCC 3.3
 Hardware: AT90S8515 at 4 Mhz, STK200 compatible starter kit

*****************************************************************************/

/* $Id: ioutil.h,v 1.9 2008/11/26 21:13:01 awachtler Exp $ */
/**
 * @file
 * @brief Interface for @ref grpLibIoutil.
 */

#ifndef IOUTIL_H
#define IOUTIL_H

/* === Includes ============================================================== */
#include "board.h"
#include "hif.h"
#include "timer.h"

/* === Externals ============================================================= */

/* === Types ================================================================= */

/* === Macros ================================================================ */

/* === LED Handling === */

/** @ingroup grpIoUtilLed */
/** @{ */

#if defined(NO_LEDS) || defined (DOXYGEN)
# undef LED_NUMBER
  /** Number of LEDs for this board */
# define LED_NUMBER    (0)
#endif

#if defined(NO_LEDS) || defined (DOXYGEN)
  /** Initialisation of the LED port. */
# define LED_INIT() do{}while(0)
#elif !defined(LED_INIT)
# if LEDS_INVERSE == 0
#  define LED_INIT() LED_DDR |= (LED_MASK); LED_PORT &= ~(LED_MASK)
# else
#   define LED_INIT() LED_DDR |= (LED_MASK); LED_PORT |= (LED_MASK)
# endif
#endif /* !defined(LED_INIT)*/

#if defined(NO_LEDS) || defined (DOXYGEN)
  /** Display a numeric value on the LED port.
   * The value x is masked out according @ref LED_MASK,
   * so that depending on the number of LEDs of the board
   * the maximum displayed value is @ref LED_MAX_VALUE.
   */
# define LED_SET_VALUE(x) do{}while(0)
#elif !defined(LED_SET_VALUE)
# if LEDS_INVERSE == 0
#  define LED_SET_VALUE(x) \
            do {\
                LED_PORT = (LED_PORT & ~LED_MASK) | ((x<<LED_SHIFT) & LED_MASK);\
            }while(0)
# else
#  define LED_SET_VALUE(x) do {\
            LED_PORT = (LED_PORT & ~LED_MASK) | ((~x<<LED_SHIFT) & LED_MASK);\
            }while(0)
# endif
#endif /* !defined(LED_SET_VALUE)*/

#if defined(NO_LEDS) || defined (DOXYGEN)
  /** Read back the current numeric value from the LED port. */
# define LED_GET_VALUE() 0
#elif !defined(LED_GET_VALUE)
# if LEDS_INVERSE == 0
#  define LED_GET_VALUE()  ((LED_PORT & LED_MASK) >> LED_SHIFT)
# else
#  define LED_GET_VALUE()  ((~LED_PORT & LED_MASK) >> LED_SHIFT)
# endif
#endif /* !defined(LED_GET_VALUE)*/


#if defined(NO_LEDS) || defined (DOXYGEN)
  /** Switch the LED with the number  @e ln ON. */
# define LED_SET(ln) do{}while(0)
#elif !defined(LED_SET)
# if LEDS_INVERSE == 0
#  define LED_SET(ln)      LED_PORT |= (_BV(ln+LED_SHIFT) & LED_MASK)
# else
#  define LED_SET(ln)      LED_PORT &= ~(_BV(ln+LED_SHIFT) & LED_MASK)
# endif
#endif /* !defined(LED_SET)*/


#if defined(NO_LEDS) || defined (DOXYGEN)
  /** Switch the LED with the number  @e ln OFF. */
# define LED_CLR(ln) do{}while(0)
#elif !defined(LED_CLR)
# if LEDS_INVERSE == 0
#  define LED_CLR(ln)      LED_PORT &= ~(_BV(ln+LED_SHIFT) & LED_MASK)
# else
#  define LED_CLR(ln)      LED_PORT |= (_BV(ln+LED_SHIFT) & LED_MASK)
# endif
#endif /* !defined(LED_CLR)*/

#if defined(NO_LEDS) || defined (DOXYGEN)
  /** ..... */
# define LED_VAL(msk,val) do{}while(0)
#elif !defined(LED_VAL)
# if LEDS_INVERSE == 0
#  define LED_VAL(msk,val) LED_PORT |= ((LED_MASK|msk) << LED_SHIFT); \
                           LED_PORT |= ~((val << LED_SHIFT)& (LED_MASK|(msk<<LED_SHIFT)) )
# else
#  define LED_VAL(msk,val) LED_PORT &= ~(LED_MASK|(msk<<LED_SHIFT)); LED_PORT |= ~(val & (LED_MASK|msk))

# endif
#endif /* !defined(LED_VAL)*/

#if defined(NO_LEDS) || defined (DOXYGEN)
  /** Toggle the LED with the number  @e n. */
# define LED_TOGGLE(ln) do{}while(0)
#elif !defined(LED_TOGGLE)
# define LED_TOGGLE(ln) LED_PORT ^= (_BV(ln+LED_SHIFT) & LED_MASK)
#endif /* !defined(LED_TOGGLE)*/

/** Maximum value, that can be displayed on the LEDs */
#define LED_MAX_VALUE ((1<<LED_NUMBER)-1)

/** @} */


/* === KEY Handling === */

/** @ingroup grpIoUtilKey
 * @{
 */
#if defined(NO_KEYS) || defined (DOXYGEN)
  /** Initialisation of the KEY port */
# define KEY_INIT()

  /** Reading of the KEY port directly and return the value LSB aligbed. */
# define KEY_GET() (0)

#else /* defined(NO_KEYS) || defined (DOXYGEN) */
# if PULLUP_KEYS != 0
#  define PULL_MASK (MASK_KEY)
# else /* PULLUP_KEYS != 0 */
#  define PULL_MASK (0)
# endif /* PULLUP_KEYS != 0 */
# if !defined KEY_INIT
#  define KEY_INIT()  do{PORT_KEY |= PULL_MASK; DDR_KEY &= ~(MASK_KEY); }while(0)
# endif /*!defined KEY_INIT()*/
# if INVERSE_KEYS == 0
#  define KEY_GET()\
                ((PIN_KEY & MASK_KEY) >> SHIFT_KEY)
# else /* INVERSE_KEYS == 0 */
#  define KEY_GET()\
                ((~PIN_KEY & MASK_KEY) >> SHIFT_KEY)
# endif /* INVERSE_KEYS == 0 */
#endif /* defined(NO_KEYS) || defined (DOXYGEN) */
/** @} */



/**
 * @brief Debounce Key values, returned from the macro KEY_GET()
 * @return status ot the debounced key
 * @ingroup grpIoUtilKey
 */
static inline uint8_t keys_debounced(void)
{
  static uint8_t key_state;		// debounced and inverted key state:
  static uint8_t ct0, ct1;      // holds two bit counter for each key
  uint8_t i;


  /*
   * read current state of keys (active-low),
   * clear corresponding bit in i when key has changed
   */
  i = key_state ^ KEY_GET();   // key changed ?

  /*
   * ct0 and ct1 form a two bit counter for each key,
   * where ct0 holds LSB and ct1 holds MSB
   * After a key is pressed longer than four times the
   * sampling period, the corresponding bit in key_state is set
   */
  ct0 = ~( ct0 & i );			// reset or count ct0
  ct1 = (ct0 ^ ct1) & i;	    // reset or count ct1
  i &= ct0 & ct1;			    // count until roll over ?
  key_state ^= i;			    // then toggle debounced state

  /*
   * To notify main program of pressed key, the correspondig bit
   * in global variable key_press is set.
   * The main loop needs to clear this bit
   */
  return key_state & i;	// 0->1: key press detect

}

/**
 * @brief Debounce Key values, returned from the macro KEY_GET()
 * @return status ot the debounced key
 * @ingroup grpIoUtilKey
 */
static inline void trap_if_key_pressed(void)
{

    KEY_INIT();
    DELAY_MS(10);
    if (KEY_GET())
    {
        LED_INIT();
        while(1)
        {
            DELAY_MS(400);
            LED_SET(0);
            DELAY_MS(10);
            LED_CLR(0);
        }
    }
}
/* === Bootloader Interface === */
#if BOOT_LOADER_ADDRESS != 0

#define JUMP_BOOT_LOADER() \
    do {\
        void (*funcptr)( uint8_t flag ) = BOOT_LOADER_ADDRESS;\
        funcptr(42);\
    }while(0)
#else /* BOOT_LOADER_ADDRESS != 0 */
#define JUMP_BOOT_LOADER()
#endif /* BOOT_LOADER_ADDRESS != 0 */




/* === Prototypes ============================================================ */
#ifdef __cplusplus
extern "C" {
#endif

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif  /* #ifndef IOUTIL_H */
/* EOF */
