/* Copyright (c) 2010 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: wibohost.c,v 1.18 2011/01/09 14:23:18 awachtler Exp $ */
/**
 * @file
 * @brief Wireless bootloader application, resides in bootloader section
 *
 * @author Daniel Thiele
 *         Dietzsch und Thiele PartG
 *         Bautzner Str. 67 01099 Dresden
 *         www.ib-dt.de daniel.thiele@ib-dt.de
 *
 * @ingroup grpAppWiBo
 */
#include <util/crc16.h>
#include <board.h>
#include <transceiver.h>
#include <radio.h>
#include "wibocmd.h"
#include "wibohost.h"

#ifndef NODEADDR_STATIC
#define NODEADDR_STATIC (0)
#endif

#if NODEADDR_STATIC > 0
#ifndef PANID
#define PANID (0x0001)
#endif
#ifndef SHORTADDR
#define SHORTADDR (0x0020)
#endif
#ifndef LONGADDR
#define LONGADDR (0x0000)
#endif
#endif

/* collect data to send here */
static uint8_t txbuf[MAX_FRAME_SIZE];

/* running checksum to calculate over all send data packets (via feed())
 * to be set to zero on start of data stream (reset())
 */
static uint16_t datacrc = 0x0000;

/* frame receive buffer */
static uint8_t rxbuf[MAX_FRAME_SIZE];

static node_config_t nodecfg;

static volatile uint8_t pingreplied = 0;

/** forward declarations */
void cb_wibohost_pingreply(wibocmd_pingreply_t *pr);

/******************* uracoli callback functions ***************/

node_config_t* wibohost_getnodecfg()
{
	return &nodecfg;
}

void usr_radio_error(radio_error_t err)
{
}

void usr_radio_irq(uint8_t cause)
{
}

uint8_t * usr_radio_receive_frame(uint8_t len, uint8_t *frm, uint8_t lqi,
		int8_t ed, uint8_t crc_fail)
{
	wibocmd_pingreply_t *pr = (wibocmd_pingreply_t*) frm;

	/* decode command code */
	if (WIBOCMDCODE_PINGREPLY == pr->hdr.command) {
		cb_wibohost_pingreply(pr);
	} else {
		/* unknown command, skip the frame */
	}

	return frm;
}

void usr_radio_tx_done(radio_tx_done_t status)
{
	if (status == TX_OK) {
	} else {
		/* TODO handle error */
	}
}

void wibohost_init(void)
{
#if NODEADDR_STATIC > 0
	nodecfg.pan_id = PANID;
	nodecfg.short_addr = SHORTADDR;
	nodecfg.ieee_addr = LONGADDR;
#else
	get_node_config(&nodecfg);
#endif
	radio_init(rxbuf, MAX_FRAME_SIZE);
	radio_set_param(RP_PANID(nodecfg.pan_id));
	radio_set_param(RP_SHORTADDR(nodecfg.short_addr));
	radio_set_param(RP_IDLESTATE(STATE_RXAUTO));
}

/*
 * \brief Send a command over the air
 *
 * Construct the header that is the same for all commands
 */
static void wibohost_sendcommand(uint16_t short_addr, wibocmdcode_t cmdcode,
		uint8_t *data, uint8_t lendata)
{
	wibocmd_hdr_t *hdr = (wibocmd_hdr_t*) data;

	hdr->FCF = 0x8801; /* short addressing, frame type: data, no ACK requested */
	hdr->command = cmdcode;
	hdr->destpanid = nodecfg.pan_id;
	hdr->destaddr = short_addr;
	hdr->srcpanid = nodecfg.pan_id;
	hdr->srcaddr = nodecfg.short_addr;
	radio_set_state(STATE_TXAUTO);
	radio_send_frame(lendata + 2, data, 1); /* +2: add CRC bytes */
}

void wibohost_feed(uint16_t short_addr, uint8_t *data, uint8_t lendata)
{
	wibocmd_data_t *dat = (wibocmd_data_t*) txbuf;
	uint8_t i;

	for (i = 0; i < lendata; i++) {
		datacrc = _crc_ccitt_update(datacrc, data[i]);
		dat->data[i] = data[i]; /* no use for memcpy() */
	}
	dat->dsize = lendata;
	dat->targmem = 'F';

	wibohost_sendcommand(short_addr, WIBOCMDCODE_DATA, (uint8_t*) dat,
			sizeof(wibocmd_data_t) + lendata);
}

void wibohost_ping(uint16_t short_addr)
{
	wibocmd_ping_t *ping = (wibocmd_ping_t*) txbuf;

	/* no data fields to fill */

	wibohost_sendcommand(short_addr, WIBOCMDCODE_PING, (uint8_t*) ping,
			sizeof(wibocmd_ping_t));
}

uint8_t wibohost_pingreplied(void)
{
	return pingreplied;
}

/*
 * \brief Called asynchronous when ping reply frame is received
 */
void cb_wibohost_pingreply(wibocmd_pingreply_t *pr)
{
	printf_P(
			PSTR("OK {'nodeid':0x%02X, 'boardname':'%s', 'swversion':0x%02X, 'crc':0x%04X, 'pagesize':0x%X}\n"),
			pr->hdr.srcaddr, pr->boardname, pr->swversion, pr->crc, pr->pagesize);
	pingreplied = 1;
}

void wibohost_finish(uint16_t short_addr)
{
	wibocmd_finish_t *dat = (wibocmd_finish_t*) txbuf;

	/* no data fields to fill */

	wibohost_sendcommand(short_addr, WIBOCMDCODE_FINISH, (uint8_t*) dat,
			sizeof(wibocmd_finish_t));
}

/*
 * \brief Always reset all nodes in network because our own datacrc is reset, otherwise a separate CRC for each node
 * 		has to be hold on the host (here)
 */
void wibohost_reset(void)
{
	wibocmd_reset_t *dat = (wibocmd_reset_t*) txbuf;

	/* no data fields to fill */

	wibohost_sendcommand(0xFFFF, WIBOCMDCODE_RESET, (uint8_t*) dat,
			sizeof(wibocmd_reset_t));
	datacrc = 0x0000;
}

uint16_t wibohost_getcrc()
{
	return datacrc;
}

void wibohost_exit(uint16_t short_addr)
{
	wibocmd_exit_t *dat = (wibocmd_exit_t*) txbuf;

	/* no data fields to fill */

	wibohost_sendcommand(short_addr, WIBOCMDCODE_EXIT, (uint8_t*) dat,
			sizeof(wibocmd_exit_t));
}

void wibohost_xmpljbootl(uint16_t short_addr)
{
	wibocmd_xmpljbootl_t *dat = (wibocmd_xmpljbootl_t*) txbuf;

	wibohost_sendcommand(short_addr, WIBOCMDCODE_XMPLJBOOTL, (uint8_t*) dat,
			sizeof(wibocmd_xmpljbootl_t));
}

/* EOF */
