/* Copyright (c) 2013 - 2015 Daniel Thiele, Tom Magnier, Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

#ifndef HardwareRadio_h
#define HardwareRadio_h

/**
 * @file
 * @brief Arduino Radio Class.
 * @addtogroup grpContribArduino
 * @{
 */

#include <string.h>
#include "board.h"
#include "radio.h"
#include "p2p_protocol.h"
#include "Stream.h"

/** Minimun radio channel number in 2.4 GHz band */
#define PHY_MIN_CHANNEL (11)
/** Maximun radio channel number in 2.4 GHz band */
#define PHY_MAX_CHANNEL (26)
/** Default radio channel number in 2.4 GHz band */
#define PHY_DEFAULT_CHANNEL (17)

/** Maximum size of a IEEE 802.15.4 frame */
#define PHY_MAX_FRAME_SIZE (127)

/** Element of a chained list of frame buffers */
typedef struct radio_buffer
{
    /** pointer to next list element or NULL if list terminates */
    struct radio_buffer *next;
    /** Length of payload */
    uint8_t len;
    /** array that can store a maximum IEEE 802.15.4 frame */
    uint8_t frm[PHY_MAX_FRAME_SIZE];
    /** read/write index */
    uint8_t idx;
}
radio_buffer_t;

# if defined(__cplusplus) || defined(DOXYGEN)
/** Hardware Radio class */
class HardwareRadio : public Stream
{
  private:
    uint8_t txseq;
    void txbuffer_init(radio_buffer_t *pbuf);
    /* user reads from this buffer */
    radio_buffer_t *prdbuf;
    /* user writes to this buffer */
    radio_buffer_t *pwrbuf;

  public:
    /* buffer which is currently transmited */
    radio_buffer_t *txbuf;
    /* buffer which is currently used for reception */
    radio_buffer_t *rxbuf;

    volatile uint8_t tx_in_progress;
    node_config_t nc;

    /** Allocate a radio buffer */
    radio_buffer_t * alloc_buffer(void);

    /** Free a radio buffer */
    void free_buffer(radio_buffer_t * pbuf);


    /** constructor */
    HardwareRadio(void);

    /** Starting the hardware radio class with default parameters */
    void begin(void);

    /**  Starting the hardware radio class with explicit parameters
     * @param channel radio channel
     *          (11 - 26 for 2.4GHz radios, 0 - 10 for SubGHz radios)
     *  @param idlestate default state of the radio, supported values are listed
     *           in @ref radio_state_t.
     */
    void begin(uint8_t channel, uint8_t idlestate);

    /** return number of available bytes in current RX buffer */
    virtual int available(void);

    /** Returns the next byte (character) of incoming data (RX)
     *  without removing it from the internal serial buffer.
     *  @return EOF (-1) if no data available, otherwise a value
     *          from 0 ... 255
     */
    virtual int peek(void);
    /** Returns the next byte (character) of incoming data (RX)
     *  @return EOF (-1) if no data available, otherwise a value
     *          from 0 ... 255
     */
    virtual int read(void);
    /**
     * flush TX and RX queues.
     * RX queue data are discarded, TX data is sent.
     */
    virtual void flush(void);

    /** write a byte to the TX stream */
    virtual size_t write(uint8_t);

    #if ! defined(DOXYGEN)
    using Print::write; // pull in write(str) and write(buf, size) from Print
    #else
    /** write a string to the TX stream
     * @param str \\0 terminated string
     */
    void write(char * str);

    /** write a binary buffer (buf, size) to the TX stream
     * @param buf pointer to the buffer
     * @param size number of bytes in the buffer.
     */
    void write(uint8_t *buf, uint8_t size);
    #endif
};

#  ifndef HARDWARERADIO_CPP
    extern HardwareRadio Radio;
#  endif
# endif
 /** @} */
#endif /*#ifndef HardwareRadio_h*/
