/* Copyright (c) 2012 - 2013 Samuel Klipphahn
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: $ */
/**
 * @file
 * @brief Interface for lwMesh, routing part
 * @ingroup grplwMesh
 */
#ifndef LWROUTE_H
#define LWROUTE_H

/**
 *  @addtogroup grplwMesh
 *  @{
 */
/* === includes ============================================================ */
#include "lw_mesh.h"
#include "lw_mac.h"

/* === macros ============================================================== */

/** Is assigned to new discovered routes. Defines how many failed attempts
 *  to use this route are performed before this route is dropped. */
#define NWK_ROUTE_DEFAULT_SCORE     (3)

/** return value for @ref get_next_hop_route_id(uint16_t dst_addr) */
#define NWK_NO_ROUTE_ID             (NWK_ROUTE_TABLE_SIZE)

/** devices with short address < 0x80000 are routing nodes,
 *  devices with short address >= 0x80000 are non-routing nodes */
#define NWK_IS_ROUTER_ADDR(x)       (x < 0x80000)

/* === types =============================================================== */

/**
 * @brief Entry of routing table
 */
typedef struct
{
    uint16_t    dstAddr;        /**< short destination address */
    uint16_t    nextHopAddr;    /**< short address of next hop */
    uint8_t     score;          /**< record score to remove stale records */
    uint8_t     lqi;            /**< LQI for last received frame from
                                     nextHopAddr node */
} NWK_RouteTableEntry_t;

typedef struct
{
    uint16_t    lw_src_addr;    /**< lightweight source address of frame*/
    uint8_t     lw_seq;         /**< lightweight sequence number of frame*/
    uint16_t    ttl;            /**< time to deletion of this record (in ms)*/
} NWK_DuplicateRejectionTableEntry_t;

/* === prototypes ========================================================== */
#ifdef __cplusplus
extern "C" {
#endif

/**
 * initializes lightweight route functions (resets routing table)
 * */
void lwr_init();

/**
 * saves routing information from a received frame
 * @param[in] lw_src_addr   lightweight source address of received frame
 * @param[in] m_src_addr    MAC source address of received frame
 * @param[in] lqi           LQI value of received frame
 * */
void lwr_save_routes(uint16_t lw_src_addr, uint16_t m_src_addr, uint8_t lqi);

/**
 * searches for route to dst_addr and returns address of next node on this
 * route
 * @param[in] dst_addr  address of destination node
 * @return              address of next node on route to dst_addr or
 *                      NWK_BROADCAST_ADDR if route to dst_addr is unknown
 * */
uint16_t lwr_get_next_hop_addr(uint16_t dst_addr);

bool lwr_prep_r_frame(NWK_Tasklist_t *task);

void lwr_r_err_received(uint16_t src_addr, uint16_t dst_addr);

void lwr_data_confirm(NWK_DataReq_t *req);

bool lwr_is_duplicate_frame(uint8_t lw_seq, uint16_t lw_src_addr);

#ifdef __cplusplus
} /* extern "C" */
#endif

/**
 *  @}
 */
#endif  /* #ifndef LWROUTE_H */
